package name.remal.gradle_plugins

import name.remal.gradle_plugins.RemalGradlePlugins.ClassesRelocationPluginDisabledID
import name.remal.gradle_plugins.artifact.ArtifactsCacheCleanerPlugin
import name.remal.gradle_plugins.classes_relocation.ClassesRelocator
import name.remal.gradle_plugins.dsl.java
import name.remal.gradle_plugins.dsl.javaPackageName
import name.remal.gradle_plugins.utils.*
import name.remal.gradle_plugins.utils.PluginIds.JACOCO_PLUGIN_ID
import name.remal.gradle_plugins.utils.PluginIds.JAVA_PLUGIN_ID
import org.gradle.api.Project
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ModuleDependency
import org.gradle.api.artifacts.ResolvedArtifact
import org.gradle.api.artifacts.UnknownConfigurationException
import org.gradle.api.artifacts.component.ComponentArtifactIdentifier
import org.gradle.api.artifacts.component.ModuleComponentIdentifier
import org.gradle.api.artifacts.component.ProjectComponentIdentifier
import org.gradle.api.plugins.WarPlugin
import org.gradle.api.tasks.SourceSet
import org.gradle.api.tasks.SourceSet.MAIN_SOURCE_SET_NAME
import org.gradle.api.tasks.compile.AbstractCompile
import org.gradle.testing.jacoco.tasks.JacocoReportBase
import org.slf4j.LoggerFactory
import java.util.stream.Collectors.toSet
import java.util.stream.Stream

@API
class ClassesRelocationPlugin : ProjectPlugin() {

    companion object {
        @JvmStatic private val logger = LoggerFactory.getLogger(ClassesRelocationPlugin::class.java)
        const val RELOCATE_ALL_CLASSES_CONFIGURATION_NAME = "relocateAllClasses"
        const val RELOCATE_USED_CLASSES_CONFIGURATION_NAME = "relocateClasses"
    }

    override fun apply(project: Project) {
        if (project.isDisabledBy(ClassesRelocationPluginDisabledID)) return

        project.applyPlugin(ArtifactsCacheCleanerPlugin::class.java)

        project.withPlugin(JAVA_PLUGIN_ID) {

            val relocateAllConf = project.configurations.create(RELOCATE_ALL_CLASSES_CONFIGURATION_NAME) {
                it.description = "Dependencies to relocate all classes"
                it.isCanBeConsumed = false
            }
            val relocateUsedConf = project.configurations.create(RELOCATE_USED_CLASSES_CONFIGURATION_NAME) {
                it.description = "Dependencies to relocate only used classes"
                it.isCanBeConsumed = false
            }

            val sourceSet = project.java.sourceSets[MAIN_SOURCE_SET_NAME]
            project.configurations[sourceSet.compileOnlyConfigurationName].extendsFrom(relocateAllConf, relocateUsedConf)

            project.afterEvaluateOrdered {
                if (relocateAllConf.allDependencies.isEmpty() && relocateUsedConf.allDependencies.isEmpty()) return@afterEvaluateOrdered

                addExclusions(project, sourceSet, relocateAllConf, relocateUsedConf)

                project.tasks.configure(AbstractCompile::class.java) { task ->
                    task.doLastOrdered(Int.MAX_VALUE) {
                        if (relocateAllConf.allDependencies.isEmpty() && relocateUsedConf.allDependencies.isEmpty()) return@doLastOrdered

                        val relocateAllFiles = relocateAllConf.resolve()
                        val relocateUsedFiles = relocateUsedConf.resolve()
                        if (task.classpath.none { relocateAllFiles.contains(it) || relocateUsedFiles.contains(it) }) return@doLastOrdered

                        val relocator = ClassesRelocator(task.destinationDir, project.relocatedClassesBasePackageName, relocateAllConf, relocateUsedConf)
                        relocator.doRelocateClasses()
                    }
                }
            }

            project.withPlugin(JACOCO_PLUGIN_ID) {
                project.configureTasks(JacocoReportBase::class.java) { task ->
                    task.doFirstOrdered {
                        task.filterAllClassDirectories { it.exclude(project.relocatedClassesBasePackageName.replace('.', '/') + "/**") }
                    }
                }
            }

        }
    }

    private fun addExclusions(project: Project, sourceSet: SourceSet, relocateAllConf: Configuration, relocateUsedConf: Configuration) {
        val flatRelocateConfs = Stream.of(relocateAllConf, relocateUsedConf)
            .flatMap { it.hierarchy.stream() }
            .collect(toSet())

        val exclusionConfs = Stream.of(
            sourceSet.compileClasspathConfigurationName,
            sourceSet.runtimeClasspathConfigurationName,
            WarPlugin.PROVIDED_RUNTIME_CONFIGURATION_NAME
        )
            .flatMap {
                try {
                    project.configurations[it].hierarchy.stream()
                } catch (e: UnknownConfigurationException) {
                    Stream.of<Configuration>()
                }
            }
            .filter { project.configurations.contains(it) }
            .filter { !flatRelocateConfs.contains(it) }
            .collect(toSet())

        logger.debug("Exclusion configurations: {}", exclusionConfs)

        addExclusions(project, relocateAllConf.hierarchy, exclusionConfs)

        addExclusions(project, relocateUsedConf.hierarchy, exclusionConfs)
        addExclusions(project, relocateUsedConf.hierarchy, relocateAllConf.hierarchy)
    }

    private fun addExclusions(project: Project, relocateConfs: Set<Configuration>, exclusionConfs: Set<Configuration>) {
        exclusionConfs.stream()
            .filter(Configuration::isCanBeResolved)
            .filter { !it.dependencies.isEmpty() }
            .map(Configuration::copy)
            .flatMap { it.resolvedConfiguration.resolvedArtifacts.stream() }
            .map(ResolvedArtifact::getId)
            .map(ComponentArtifactIdentifier::getComponentIdentifier)
            .map { id ->
                if (id is ModuleComponentIdentifier) {
                    mapOf(
                        "group" to id.group,
                        "module" to id.module
                    )
                } else if (id is ProjectComponentIdentifier) {
                    val projectPath = id.projectPath
                    val dependencyProject = project.rootProject.evaluationDependsOn(projectPath)
                    mapOf(
                        "group" to dependencyProject.group.toString(),
                        "module" to dependencyProject.name
                    )
                } else {
                    mapOf()
                }
            }
            .filter { it.isNotEmpty() }
            .distinct()
            .forEach { excludeProperties ->
                relocateConfs.forEach { conf ->
                    logger.debug("Excluding {} from {}", excludeProperties, conf)
                    conf.dependencies.all { dep ->
                        if (dep is ModuleDependency) dep.exclude(excludeProperties)
                    }
                }
            }
    }

}


val Project.relocatedClassesBasePackageName: String get() = this.javaPackageName + "._"
