package name.remal.gradle_plugins

import name.remal.gradle_plugins.RemalGradlePlugins.FlattenJavaProjectPluginDisabledID
import name.remal.gradle_plugins.dsl.getOrCreateSourcesJarTask
import name.remal.gradle_plugins.dsl.java
import name.remal.gradle_plugins.dsl.makeSrcDirsUnique
import name.remal.gradle_plugins.utils.*
import name.remal.gradle_plugins.utils.PluginIds.JAVA_PLUGIN_ID
import org.gradle.api.Project
import org.gradle.api.artifacts.ProjectDependency
import org.gradle.api.file.CopySpec
import org.gradle.api.file.FileTree
import org.gradle.api.plugins.JavaPlugin.JAR_TASK_NAME
import org.gradle.api.tasks.SourceSet.MAIN_SOURCE_SET_NAME
import org.gradle.api.tasks.bundling.AbstractArchiveTask
import java.util.concurrent.Callable

@API
class FlattenJavaProjectPlugin : ProjectPlugin() {

    override fun apply(project: Project) {
        if (project.isDisabledBy(FlattenJavaProjectPluginDisabledID)) return

        project.evaluationDependsOnChildren()

        project.applyPlugin(JAVA_PLUGIN_ID)
        project.applyPlugin(MergeJavaResourcesPlugin::class.java)

        this.setupConfigurations(project)
        this.setupTasks(project)
    }

    private fun setupConfigurations(project: Project) {
        val subprojectsToFlatten = project.subprojectsToFlatten

        val mainSourceSet = project.java.sourceSets[MAIN_SOURCE_SET_NAME]
        arrayOf(
            mainSourceSet.compileConfigurationName,
            mainSourceSet.apiConfigurationName,
            mainSourceSet.apiElementsConfigurationName,
            mainSourceSet.implementationConfigurationName,
            mainSourceSet.runtimeConfigurationName
        ).forEach { confName ->

            project.configurations.configure(confName) { conf ->
                subprojectsToFlatten.forEach { subproject ->
                    subproject.configurations.findByName(conf.name)?.let {
                        it.dependencies.all {
                            if (it is ProjectDependency && it.dependencyProject in subprojectsToFlatten) {
                                // do nothing
                            } else {
                                conf.dependencies.add(it)
                            }
                        }
                    }
                }
            }

        }
    }

    private fun setupTasks(project: Project) {
        fun CopySpec.addCommonExcludes() {
            exclude("META-INF/*.MF")
            exclude("META-INF/*.SF")
            exclude("META-INF/*.DSA")
            exclude("META-INF/*.RSA")
        }

        val jar = project.tasks[JAR_TASK_NAME] as AbstractArchiveTask
        project.subprojectsToFlatten.forEach { subproject ->
            subproject.tasks.findByName(jar.name)?.let {
                jar.dependsOn(it)

                it as AbstractArchiveTask
                jar.from(Callable<FileTree> { project.zipTree(it.archivePath) }, CopySpec::addCommonExcludes)
            }
        }


        val sourcesJar = project.getOrCreateSourcesJarTask()
        project.subprojectsToFlatten.forEach { subproject ->
            sourcesJar.from(subproject.java.sourceSets[MAIN_SOURCE_SET_NAME].allSource.makeSrcDirsUnique())
        }
    }

    private val Project.subprojectsToFlatten: Set<Project>
        get() = this.subprojects.asSequence()
            .filter { it.pluginManager.hasPlugin(JAVA_PLUGIN_ID) }
            .filter { !it.isDisabledBy(FlattenJavaProjectPluginDisabledID) }
            .toSet()

}
