package name.remal.gradle_plugins

import name.remal.gradle_plugins.RemalGradlePlugins.IntegrationTestingPluginDisabledID
import name.remal.gradle_plugins.dsl.java
import name.remal.gradle_plugins.utils.*
import name.remal.gradle_plugins.utils.PluginIds.JACOCO_PLUGIN_ID
import name.remal.gradle_plugins.utils.PluginIds.JAVA_PLUGIN_ID
import org.apache.commons.lang.StringUtils.capitalize
import org.gradle.api.Project
import org.gradle.api.plugins.JavaPlugin.TEST_TASK_NAME
import org.gradle.api.reporting.Report
import org.gradle.api.tasks.SourceSet.MAIN_SOURCE_SET_NAME
import org.gradle.api.tasks.SourceSet.TEST_SOURCE_SET_NAME
import org.gradle.api.tasks.testing.Test
import org.gradle.language.base.plugins.LifecycleBasePlugin.VERIFICATION_GROUP
import org.gradle.testing.jacoco.plugins.JacocoPluginExtension
import org.gradle.testing.jacoco.plugins.JacocoTaskExtension
import org.gradle.testing.jacoco.tasks.JacocoCoverageVerification
import org.gradle.testing.jacoco.tasks.JacocoReport
import org.gradle.testing.jacoco.tasks.JacocoReportBase
import java.io.File

@API
class IntegrationTestingPlugin : ProjectPlugin() {

    companion object {
        const val INTEGRATION_SOURCE_SET_NAME = "integration"
        const val INTEGRATION_TEST_TASK_NAME = "integrationTest"
        const val ALL_TESTS_TASK_NAME = "allTests"
    }

    override fun apply(project: Project) {
        if (project.isDisabledBy(IntegrationTestingPluginDisabledID)) return

        project.withPlugin(JAVA_PLUGIN_ID) {

            val integrationSourceSet = project.java.sourceSets.create(INTEGRATION_SOURCE_SET_NAME) { sourceSet ->
                sourceSet.java.srcDir(project.file("src/${sourceSet.name}/java"))
                sourceSet.resources.srcDir(project.file("src/${sourceSet.name}/resources"))
            }

            project.dependencies.apply {
                val mainSourceSet = project.java.sourceSets[MAIN_SOURCE_SET_NAME]
                val testSourceSet = project.java.sourceSets[TEST_SOURCE_SET_NAME]

                arrayOf(
                    project.configurations[testSourceSet.compileConfigurationName]
                    , mainSourceSet.output
                ).forEach {
                    add(integrationSourceSet.compileConfigurationName, it)
                }

                arrayOf(
                    project.configurations[testSourceSet.runtimeConfigurationName]
                ).forEach {
                    add(integrationSourceSet.runtimeConfigurationName, it)
                }
            }

            val integrationTestTask = project.tasks.create(INTEGRATION_TEST_TASK_NAME, Test::class.java) { task ->
                task.group = VERIFICATION_GROUP
                task.description = "Runs the integration tests"
                task.testClassesDirs = integrationSourceSet.output.classesDirs
                task.classpath = integrationSourceSet.runtimeClasspath
            }

            project.tasks.create(ALL_TESTS_TASK_NAME) { task ->
                task.group = integrationTestTask.group
                task.description = "Runs all tests"
                task.dependsOn(project.tasks[TEST_TASK_NAME])
                task.dependsOn(integrationTestTask)
            }

            project.withPlugin(JACOCO_PLUGIN_ID) withJacocoPlugin@ {
                if (JacocoTaskExtension::class.java in integrationTestTask.extensions) return@withJacocoPlugin

                val extension = project[JacocoPluginExtension::class.java]
                extension.applyTo(integrationTestTask)

                fun JacocoReportBase.baseSetup() {
                    group = VERIFICATION_GROUP
                    executionData(integrationTestTask)
                    sourceSets(integrationSourceSet)
                }

                project.tasks.create("jacoco" + capitalize(integrationTestTask.name) + "Report", JacocoReport::class.java) { task ->
                    task.description = "Generates code coverage report for the ${integrationTestTask.name} task."
                    task.baseSetup()
                    task.reports.configure {
                        if (Report.OutputType.DIRECTORY == it.outputType) {
                            it.setDestination(project.provider({ File(extension.reportsDir, integrationTestTask.name + "/" + it.name) }))
                        } else {
                            it.setDestination(project.provider({ File(extension.reportsDir, integrationTestTask.name + "/" + task.name + "." + it.name) }))
                        }
                    }
                }

                project.tasks.create("jacoco" + capitalize(integrationTestTask.name) + "CoverageVerification", JacocoCoverageVerification::class.java) { task ->
                    task.description = "Verifies code coverage metrics based on specified rules for the ${integrationTestTask.name} task."
                    task.baseSetup()
                }
            }

        }
    }

}
