package name.remal.gradle_plugins

import name.remal.gradle_plugins.RemalGradlePlugins.JUnitPlatformSettingsPluginDisabledID
import name.remal.gradle_plugins.dsl.getJunitPlatformTestTasks
import name.remal.gradle_plugins.dsl.java
import name.remal.gradle_plugins.utils.*
import name.remal.gradle_plugins.utils.PluginIds.JACOCO_PLUGIN_ID
import name.remal.gradle_plugins.utils.PluginIds.JUNIT_PLATFORM_PLUGIN_ID
import org.apache.commons.lang.StringUtils.capitalize
import org.gradle.api.Project
import org.gradle.api.reporting.Report
import org.gradle.api.tasks.JavaExec
import org.gradle.api.tasks.SourceSet.MAIN_SOURCE_SET_NAME
import org.gradle.api.tasks.testing.Test
import org.gradle.language.base.plugins.LifecycleBasePlugin.VERIFICATION_GROUP
import org.gradle.testing.jacoco.plugins.JacocoPluginExtension
import org.gradle.testing.jacoco.plugins.JacocoTaskExtension
import org.gradle.testing.jacoco.tasks.JacocoCoverageVerification
import org.gradle.testing.jacoco.tasks.JacocoReport
import org.gradle.testing.jacoco.tasks.JacocoReportBase
import org.junit.platform.gradle.plugin.JUnitPlatformExtension
import java.io.File

@API
class JUnitPlatformSettingsPlugin : ProjectPlugin() {

    override fun apply(project: Project) {
        if (project.isDisabledBy(JUnitPlatformSettingsPluginDisabledID)) return

        project.withPlugin(JUNIT_PLATFORM_PLUGIN_ID) {
            val extension = project[JUnitPlatformExtension::class.java].also { ext ->
                ext.configurationParameter("junit.jupiter.extensions.autodetection.enabled", "true")
            }

            project.withPlugin(JACOCO_PLUGIN_ID) {
                project.tasks.getJunitPlatformTestTasks().configure { testTask ->
                    if (JacocoTaskExtension::class.java in testTask.extensions) return@configure

                    when (testTask) {
                        is JavaExec -> project[JacocoPluginExtension::class.java].applyTo(testTask)
                        is Test -> project[JacocoPluginExtension::class.java].applyTo(testTask)
                        else -> return@configure
                    }

                    fun JacocoReportBase.baseSetup() {
                        group = VERIFICATION_GROUP
                        executionData(testTask)
                        sourceSets(project.java.sourceSets[MAIN_SOURCE_SET_NAME])
                    }

                    project.tasks.create("jacoco" + capitalize(testTask.name) + "Report", JacocoReport::class.java) { task ->
                        task.description = "Generates code coverage report for the ${testTask.name} task."
                        task.baseSetup()
                        task.reports.configure {
                            if (Report.OutputType.DIRECTORY == it.outputType) {
                                it.setDestination(project.provider({ File(extension.reportsDir, testTask.name + "/" + it.name) }))
                            } else {
                                it.setDestination(project.provider({ File(extension.reportsDir, testTask.name + "/" + task.name + "." + it.name) }))
                            }
                        }
                    }

                    project.tasks.create("jacoco" + capitalize(testTask.name) + "CoverageVerification", JacocoCoverageVerification::class.java) { task ->
                        task.description = "Verifies code coverage metrics based on specified rules for the ${testTask.name} task."
                        task.baseSetup()
                    }
                }
            }
        }
    }

}
