package name.remal.gradle_plugins

import com.google.common.collect.MultimapBuilder
import name.remal.gradle_plugins.RemalGradlePlugins.MergeJavaResourcesPluginDisabledID
import name.remal.gradle_plugins.dsl.*
import name.remal.gradle_plugins.utils.*
import name.remal.gradle_plugins.utils.Constants.SERVICE_FILE_BASE_PATH
import name.remal.gradle_plugins.utils.Constants.TEXT_FILE_CHARSET
import name.remal.gradle_plugins.utils.PluginIds.JAVA_PLUGIN_ID
import org.gradle.api.Project
import org.gradle.api.file.RelativePath
import org.gradle.api.internal.file.TemporaryFileProvider
import org.gradle.api.tasks.AbstractCopyTask
import java.io.File
import java.util.*
import javax.inject.Inject

@API
class MergeJavaResourcesPlugin @Inject constructor(
    private val temporaryFileProvider: TemporaryFileProvider
) : ProjectPlugin() {

    override fun apply(project: Project) {
        if (project.isDisabledBy(MergeJavaResourcesPluginDisabledID)) return

        project.withPlugin(JAVA_PLUGIN_ID) {
            setupServicesMerging(project)
        }
    }

    private fun setupServicesMerging(project: Project) {
        val servicesBaseSegments = SERVICE_FILE_BASE_PATH.split('/').toTypedArray()
        val groovyExtensionModuleService = "org.codehaus.groovy.runtime.ExtensionModule"

        project.configureTasks(AbstractCopyTask::class.java) { task ->

            task.doFirstOrdered {
                val mergedFilesDir = temporaryFileProvider.newTemporaryFile(task.name + ".merged-services").forceDelete()

                val serviceFilesMapping = MultimapBuilder.treeKeys().linkedHashSetValues().build<RelativePath, File>()
                task.rootSpec.buildRootResolver().allSource.visit { details ->
                    if (details.isDirectory) return@visit
                    details.relativePath.let { relativePath ->
                        val segments = relativePath.segments
                        if (segments.startsWith(*servicesBaseSegments) && 3 == segments.size) {
                            serviceFilesMapping.put(relativePath, details.file)
                        }
                    }
                }

                if (serviceFilesMapping.isEmpty || serviceFilesMapping.asMap().values.all { it.size <= 1 }) return@doFirstOrdered

                val serviceFiles = serviceFilesMapping.values()
                task.exclude { it.file in serviceFiles }

                val services = MultimapBuilder.treeKeys().linkedHashSetValues().build<String, String>()
                val groovyExtensionModuleProperties = mutableListOf<Properties>()
                serviceFiles.forEach { serviceFile ->
                    val service = serviceFile.name
                    if (groovyExtensionModuleService == service) {
                        groovyExtensionModuleProperties.add(loadProperties(serviceFile))
                    } else {
                        serviceFile.forEachLine(TEXT_FILE_CHARSET) { line ->
                            line.substringBefore('#').trim().nullIfEmpty()?.let { impl ->
                                services[service] = impl
                            }
                        }
                    }
                }

                if (!services.isNotEmpty || groovyExtensionModuleProperties.isNotEmpty()) {
                    task.from(mergedFilesDir)
                }

                for (serviceName in services.keySet()) {
                    val targetFile = mergedFilesDir.resolve("$SERVICE_FILE_BASE_PATH/$serviceName")
                    targetFile.createParentDirectories().writer(TEXT_FILE_CHARSET).use { writer ->
                        services.get(serviceName).forEachIndexed { index, impl ->
                            if (1 <= index) writer.append("\n")
                            writer.append(impl)
                        }
                    }
                }

                if (groovyExtensionModuleProperties.isNotEmpty()) {
                    val mergedProps = Properties()
                    groovyExtensionModuleProperties.forEach { prop ->
                        prop.forEach forEachProp@ { key, value ->
                            if (null == key || null == value) return@forEachProp
                            if (key !in mergedProps) {
                                mergedProps[key] = value.toString()
                            } else {
                                mergedProps[key] = mergedProps[key].toString() + "," + value.toString()
                            }
                        }
                    }
                    mergedProps["moduleName"] = project.id
                    mergedProps["moduleVersion"] = project.version

                    val targetFile = mergedFilesDir.resolve("$SERVICE_FILE_BASE_PATH/$groovyExtensionModuleService")
                    mergedProps.store(targetFile.createParentDirectories())
                }
            }

        }
    }

}
