package name.remal.gradle_plugins.artifact

import java.io.BufferedInputStream
import java.io.File
import java.io.FileNotFoundException
import java.io.InputStream
import java.util.*
import java.util.zip.ZipException
import java.util.zip.ZipFile

internal data class Artifact(private val file: File) : BaseHasEntries() {

    override val entryNames: Set<String> by lazy(LazyThreadSafetyMode.NONE) {
        val result = TreeSet<String>()
        if (file.isFile) {
            try {
                ZipFile(file).use { zipFile ->
                    val entries = zipFile.entries()
                    while (entries.hasMoreElements()) {
                        val entry = entries.nextElement()
                        if (entry.isDirectory) continue
                        result += entry.name
                    }
                }

            } catch (e: ZipException) {
                // do nothing
            }

        } else if (file.isDirectory) {
            file.walk().filter(File::isFile).forEach { resourceFile ->
                result += resourceFile.relativeTo(file).path.replace(File.separatorChar, '/')
            }
        }
        return@lazy result
    }

    override fun openStream(entryName: String): InputStream {
        if (file.isFile) {
            val zipFile = ZipFile(file)
            val entry = zipFile.getEntry(entryName) ?: throw ArtifactEntryNotFoundException("Artifact entry not found: $entryName")
            val inputStream = zipFile.getInputStream(entry)
            return object : BufferedInputStream(inputStream) {
                override fun close() {
                    super.close()
                    zipFile.close()
                }
            }

        } else if (file.isDirectory) {
            try {
                return File(file, entryName).inputStream()
            } catch (e: FileNotFoundException) {
                throw ArtifactEntryNotFoundException("Artifact entry not found: $entryName", e)
            }

        } else {
            throw ArtifactFileNotFoundException("Artifact file not found: $file")
        }
    }

}
