package name.remal.gradle_plugins.dsl

import com.google.common.base.CaseFormat
import name.remal.gradle_plugins.utils.Constants.ENCODING
import name.remal.gradle_plugins.utils.Constants.TEXT_FILE_CHARSET
import java.io.File
import java.io.IOException
import java.io.StringWriter
import java.net.HttpURLConnection
import java.net.URLDecoder
import java.net.URLEncoder
import java.nio.charset.Charset
import java.nio.file.FileVisitResult
import java.nio.file.Files
import java.nio.file.Path
import java.nio.file.SimpleFileVisitor
import java.nio.file.attribute.BasicFileAttributes
import java.time.Duration
import java.util.*
import java.util.stream.Stream
import java.util.stream.StreamSupport
import java.util.zip.ZipException
import java.util.zip.ZipFile

val DEFAULT_IO_TIMEOUT = Duration.ofSeconds(5)!!

fun isZipFile(file: File): Boolean {
    if (file.isFile) {
        try {
            ZipFile(file).use {
                it.entries()
                return true
            }
        } catch (e: ZipException) {
            return false
        }
    }
    return false
}

fun String?.default(value: String = ""): String = this ?: value
fun String?.nullIfEmpty(): String? = if (null == this || this.isEmpty()) null else this
fun String.encodeURIComponent(): String = URLEncoder.encode(this, ENCODING)
fun String.decodeURIComponent(): String = URLDecoder.decode(this, ENCODING)
fun String.caseFormat(from: CaseFormat, to: CaseFormat): String = from.to(to, this)

fun String.escapeRegex(): String {
    var result = this
    "-[]{}()<>*+?.,^\$|#\\".forEach {
        result = result.replace("$it", "\\$it")
    }
    return result
}

fun String.escapeJava(): String {
    @Suppress("DEPRECATION")
    return org.apache.commons.lang3.StringEscapeUtils.escapeJava(this)
}

fun StringBuilder.clear(): StringBuilder = this.apply { setLength(0) }

fun File.forceDelete() = apply {
    if (this.exists()) Files.walkFileTree(this.absoluteFile.toPath(), object : SimpleFileVisitor<Path>() {
        override fun visitFile(file: Path, attrs: BasicFileAttributes): FileVisitResult {
            Files.deleteIfExists(file)
            return FileVisitResult.CONTINUE
        }

        override fun postVisitDirectory(dir: Path, exc: IOException?): FileVisitResult {
            Files.deleteIfExists(dir)
            return FileVisitResult.CONTINUE
        }
    })
}

fun File.createDirectories() = apply {
    Files.createDirectories(this.absoluteFile.toPath())
}

fun File.createParentDirectories() = apply {
    Files.createDirectories(this.absoluteFile.parentFile.toPath())
}

fun File.forceCreate() = apply {
    try {
        Files.createFile(this.absoluteFile.createParentDirectories().toPath())
    } catch (ignored: java.nio.file.FileAlreadyExistsException) {
        // do nothing
    }
}

fun <T, R : T> Stream<T>.filterIsInstance(type: Class<R>): Stream<R> {
    return this.filter(type::isInstance).map(type::cast)
}

@JvmName("flatMapCollection")
fun <T, R> Stream<T>.flatMap(mapper: (T) -> Collection<R>): Stream<R> {
    return this.flatMap { mapper(it).stream() }
}

@JvmName("flatMapIterable")
fun <T, R> Stream<T>.flatMap(mapper: (T) -> Iterable<R>): Stream<R> {
    return this.flatMap { StreamSupport.stream(mapper(it).spliterator(), false) }
}

fun <T> Stream<T>.concatWith(otherStream: Stream<T>): Stream<T> {
    return Stream.concat(this, otherStream)
}


fun <T> Array<T>.startsWith(vararg other: T): Boolean {
    if (this.size < other.size) return false
    other.forEachIndexed { index, element ->
        if (this[index] != element) return false
    }
    return true
}


val HttpURLConnection.contentCharset: Charset
    get() {
        contentType
            ?.toUpperCase()
            ?.substringAfter("CHARSET=", "")
            ?.substringBefore(';')
            ?.trim()
            ?.let {
                try {
                    return Charset.forName(it)
                } catch (e: IllegalArgumentException) {
                    // do nothing
                }
            }

        return Charsets.UTF_8
    }

inline fun <T : AutoCloseable?, R> T.use(block: (T) -> R): R {
    var closed = false
    try {
        return block(this)

    } catch (e: Exception) {
        closed = true
        try {
            this?.close()
        } catch (closeException: Exception) {
            e.addSuppressed(closeException)
        }
        throw e

    } finally {
        if (!closed) {
            this?.close()
        }
    }
}


fun loadProperties(file: File) = Properties().apply { load(file) }

fun Properties.load(file: File) {
    file.inputStream().use { load(it) }
}

fun Properties.store(file: File, doStripComments: Boolean = true) {
    var content = StringWriter().use { store(it, ""); it.toString() }
    if (doStripComments) {
        content = content.splitToSequence('\n')
            .map { it.substringBefore('#').trim() }
            .filter(String::isNotEmpty)
            .joinToString("\n")
    }
    file.writeText(content, TEXT_FILE_CHARSET)
}
