package name.remal.gradle_plugins.embedded

import name.remal.gradle_plugins.utils.applyPlugin
import name.remal.gradle_plugins.utils.doAddURL
import name.remal.gradle_plugins.utils.forResourceStream
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.slf4j.LoggerFactory
import java.net.URLClassLoader

abstract class AbstractEmbeddedPlugin(
    private val pluginId: String,
    private val implClassName: String,
    private val resourceNames: List<String>
) : Plugin<Project> {

    private companion object {
        @JvmStatic private val logger = LoggerFactory.getLogger(AbstractEmbeddedPlugin::class.java)
        private val CLASS_LOADER: ClassLoader = AbstractEmbeddedPlugin::class.java.classLoader
    }

    override fun apply(project: Project) {
        synchronized(AbstractEmbeddedPlugin::class.java) {
            run {
                logger.debug("Trying to apply {} by class name: {}", pluginId, implClassName)
                val implClass = try {
                    @Suppress("UNCHECKED_CAST")
                    CLASS_LOADER.loadClass(implClassName) as Class<Plugin<Project>>
                } catch (ignored: ClassNotFoundException) {
                    return@run // do fallback
                }

                project.applyPlugin(implClass)
                return
            }

            run {
                logger.debug("Adding resources to class loader: {}", resourceNames)
                var classLoader = CLASS_LOADER
                while (true) {
                    if (classLoader is URLClassLoader) {
                        val urlClassLoader: URLClassLoader = classLoader
                        resourceNames.forEach { resourceName ->
                            val tempFile = run {
                                val name = resourceName.substringAfterLast('/')
                                val prefix = name.substringBeforeLast('.', "")
                                val suffix = name.substringAfterLast('.', "")
                                createTempFile(prefix + '.', '.' + suffix).absoluteFile
                            }
                            tempFile.deleteOnExit()

                            logger.debug("Extracting resource {} to {}", resourceName, tempFile)
                            forResourceStream(resourceName) { inputStream ->
                                tempFile.outputStream().use { inputStream.copyTo(it) }
                            }

                            logger.debug("Adding resource {} to class loader", resourceName)
                            urlClassLoader.doAddURL(tempFile.toURI().toURL())
                        }
                        return@run
                    }

                    classLoader = classLoader.parent ?: break
                }
            }

            logger.debug("Trying to apply {} by class name: {}", pluginId, implClassName)
            @Suppress("UNCHECKED_CAST")
            val implClass = CLASS_LOADER.loadClass(implClassName) as Class<Plugin<Project>>
            project.applyPlugin(implClass)
        }
    }

}
