package name.remal.gradle_plugins.generate_sources

import groovy.lang.Closure
import groovy.lang.DelegatesTo
import name.remal.gradle_plugins.API
import name.remal.gradle_plugins.dsl.*
import name.remal.gradle_plugins.utils.Constants.ENCODING
import name.remal.gradle_plugins.utils.delegateCall
import org.gradle.api.DefaultTask
import org.gradle.api.tasks.OutputDirectory
import org.gradle.api.tasks.TaskAction
import java.io.File
import java.io.Writer
import java.nio.charset.Charset
import java.util.function.Consumer

@API
class GenerateSourcesTask<in SourceGeneratorType : BaseSourceGenerator> : DefaultTask() {

    init {
        disableCache()
    }

    @OutputDirectory
    lateinit var outputDir: File

    lateinit var classFileExtension: String

    var textFilesEncoding: String? = null

    val textFilesCharset: Charset get() = Charset.forName(textFilesEncoding ?: ENCODING)

    @TaskAction
    protected fun createOutputDirectory() {
        outputDir.forceDelete().createDirectories()
    }

    fun addSourcesGeneration(action: (File) -> Unit) {
        doLast {
            action(outputDir)
            didWork = true
        }
    }

    fun addSourcesGeneration(action: Consumer<File>) = addSourcesGeneration(action::accept)

    fun addSourcesGeneration(@DelegatesTo(value = GenerateSourcesTask::class, strategy = Closure.DELEGATE_FIRST) action: Closure<*>) = addSourcesGeneration {
        action.delegateCall(this, it)
    }

    fun addTextResourceGeneration(fileRelativePath: String, action: (Writer) -> Unit) {
        addSourcesGeneration { outputDir ->
            val classFile = outputDir.resolve(fileRelativePath).absoluteFile
            logger.info("Generating {}", classFile)
            classFile.createParentDirectories().outputStream().bufferedWriter(textFilesCharset).use(action)
        }
    }

    fun addTextResourceGeneration(fileRelativePath: String, action: Consumer<Writer>) = addTextResourceGeneration(fileRelativePath, action::accept)

    fun addTextResourceGeneration(fileRelativePath: String, @DelegatesTo(value = GenerateSourcesTask::class, strategy = Closure.DELEGATE_FIRST) action: Closure<*>) = addTextResourceGeneration(fileRelativePath) {
        action.delegateCall(this, it)
    }

    fun addSourceGeneration(packageName: String, fileName: String, action: (Writer) -> Unit) {
        val fileRelativePath = buildString {
            if (packageName.isNotEmpty()) {
                append(packageName.replace('.', '/')).append('/')
            }
            append(fileName.replace('.', '$')).append('.').append(classFileExtension)
        }
        addTextResourceGeneration(fileRelativePath, action)
    }

    fun addSourceGeneration(fileName: String, action: (Writer) -> Unit) = addSourceGeneration(project.javaPackageName, fileName, action)

    fun addSourceGeneration(packageName: String, fileName: String, action: Consumer<Writer>) = addSourceGeneration(packageName, fileName, action::accept)

    fun addSourceGeneration(fileName: String, action: Consumer<Writer>) = addSourceGeneration(project.javaPackageName, fileName, action)

    fun addSourceGeneration(packageName: String, fileName: String, @DelegatesTo(value = GenerateSourcesTask::class, strategy = Closure.DELEGATE_FIRST) action: Closure<*>) = addSourceGeneration(packageName, fileName) {
        action.delegateCall(this, it)
    }

    fun addSourceGeneration(fileName: String, @DelegatesTo(value = GenerateSourcesTask::class, strategy = Closure.DELEGATE_FIRST) action: Closure<*>) = addSourceGeneration(project.javaPackageName, fileName, action)

    fun addSourceGeneration(generator: SourceGeneratorType) = addSourceGeneration(generator.packageName, generator.fileName) { writer ->
        generator.accept(project, writer)
    }

}
