package name.remal.gradle_plugins.utils

import io.github.lukehutch.fastclasspathscanner.FastClasspathScanner
import name.remal.gradle_plugins.utils.Constants.CLASS_FILE_NAME_SUFFIX
import java.io.File
import java.io.InputStream
import java.lang.reflect.Field
import java.net.URL
import java.net.URLClassLoader
import java.util.Collections.newSetFromMap
import java.util.concurrent.ConcurrentHashMap

private class ResourcesLoader

internal val RESOURCES_CLASS_LOADER = ResourcesLoader::class.java.classLoader

internal inline fun <T> forResourceStream(resourceName: String, action: (InputStream) -> T): T {
    val stream = RESOURCES_CLASS_LOADER.getResourceAsStream(resourceName) ?: throw IllegalStateException("$resourceName - resource not found")
    return stream.use(action)
}

internal fun classToResourcePath(className: String) = className.replace('.', '/') + CLASS_FILE_NAME_SUFFIX
internal fun classToResourcePath(clazz: Class<*>) = classToResourcePath(clazz.name)

internal fun resourcePathToClassName(path: String) = path.replace('/', '.').replace('\\', '.').trim('.').run { substring(0, length - CLASS_FILE_NAME_SUFFIX.length) }

internal inline fun forEachClassFile(classedDir: File, action: (File) -> Unit) {
    classedDir.walk()
        .filter { it.name.endsWith(CLASS_FILE_NAME_SUFFIX) }
        .filter(File::isFile)
        .forEach(action)
}

internal fun <T> concurrentSetOf(): MutableSet<T> {
    return newSetFromMap<T>(ConcurrentHashMap())
}

internal fun <T> concurrentSetOf(vararg elements: T): MutableSet<T> {
    return newSetFromMap<T>(ConcurrentHashMap()).apply {
        addAll(elements)
    }
}


internal fun getEnvOrNull(name: String, vararg fallbackNames: String): String? {
    System.getenv(name)?.let { return it }
    fallbackNames.forEach { System.getenv(it)?.let { return it } }
    return null
}

internal fun getRequiredEnv(name: String, vararg fallbackNames: String): String {
    getEnvOrNull(name, *fallbackNames)?.let { return it }
    throw IllegalStateException("Required environment variable is not set: $name (fallback: ${fallbackNames.joinToString(", ")})")
}


internal fun <C : Any, O : C> Class<C>.getField(obj: O, name: String): Any? {
    return getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.get(obj)
}

internal fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Any?) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.set(obj, value)
}

internal fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Boolean) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setBoolean(obj, value)
}

internal fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Byte) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setByte(obj, value)
}

internal fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Char) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setChar(obj, value)
}

internal fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Short) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setShort(obj, value)
}

internal fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Int) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setInt(obj, value)
}

internal fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Long) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setLong(obj, value)
}

internal fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Float) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setFloat(obj, value)
}

internal fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Double) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setDouble(obj, value)
}


internal val ClassLoader.classpathElements: List<File>
    get() = FastClasspathScanner("")
        .overrideClassLoaders(this)
        .uniqueClasspathElements.toList()

private val addURLMethod = URLClassLoader::class.java.getDeclaredMethod("addURL", URL::class.java).apply { isAccessible = true }
internal fun URLClassLoader.doAddURL(url: URL) {
    addURLMethod.invoke(this, url)
}


internal fun Class<*>.getDeclaredFieldFromHierarchy(name: String): Field {
    var currentClass = this
    while (true) {
        try {
            return currentClass.getDeclaredField(name)
        } catch (e: NoSuchFieldException) {
            if (Any::class.java == currentClass) throw e
            currentClass = currentClass.superclass
        }
    }
}

internal val Class<*>.isConcreteClass: Boolean get() = !isInterface && !isEnum && !isPrimitive && isArray && !isAnonymousClass && !isLocalClass && !isSynthetic


internal fun Char.repeat(n: Int): String {
    require(n >= 0) { "Count 'n' must be non-negative, but was $n." }
    return when (n) {
        0 -> ""
        1 -> java.lang.String.valueOf(this)
        else -> buildString {
            kotlin.repeat(n) {
                append(this)
            }
        }
    }
}
