package name.remal.gradle_plugins.utils

import name.remal.gradle_plugins.dsl.use
import name.remal.gradle_plugins.utils.Constants.ENCODING
import org.gradle.plugins.ide.idea.model.IdeaModel
import org.w3c.dom.Document
import org.w3c.dom.Element
import org.w3c.dom.Node
import java.io.File
import javax.xml.transform.OutputKeys
import javax.xml.transform.Transformer


internal fun IdeaModel.setup(action: (IdeaModel) -> Unit) {
    val ideaModel = this
    this.project.project.gradle.buildFinished {
        action(ideaModel)
    }
}

internal fun IdeaModel.setup(vararg actions: (IdeaModel) -> Unit) {
    val ideaModel = this
    this.project.project.gradle.buildFinished {
        actions.forEach { it(ideaModel) }
    }
}


internal fun IdeaModel.processConfigs(action: (Document) -> Unit) {
    this.project.project.rootProject.let { rootProject ->
        fun processXmlFile(xmlFile: File) {
            val document = parseDocument(xmlFile)
            val documentString = document.asString(false)

            action(document)

            if (documentString != document.asString(false)) {
                xmlFile.outputStream().use { outputStream ->
                    createStripSpaceTransformer().withIdeaXmlConfigSettings().transform(document, outputStream)
                }
            }
        }

        rootProject.rootDir.resolve(".idea").walk()
            .filter(File::isFile)
            .filter { it.name.endsWith(".xml") }
            .forEach(::processXmlFile)

        arrayOf("ipr", "iws").forEach { ext ->
            rootProject.rootDir.resolve("${rootProject.name}.$ext").let {
                if (it.isFile) processXmlFile(it)
            }
        }
    }
}


internal enum class IdeaConfigType {
    IPR,
    IWS,
}

internal val IDEA_CONFIG_PROJECT_VERSION = "4"

internal fun IdeaModel.writeComponentIfNotExists(componentName: String, ideaConfigType: IdeaConfigType, xmlConfigRelativePath: String, wrapperNodeCreator: (Document) -> Node = { it }) {
    this.project.project.rootProject.let { rootProject ->
        fun isXmlFileWithComponent(xmlFile: File): Boolean {
            if (!xmlFile.isFile) return false
            val document = parseDocument(xmlFile)
            return document.hasIdeaComponentElement(componentName)
        }

        run {
            val ideaDir = rootProject.rootDir.resolve(".idea")
            if (ideaDir.isDirectory) {
                val xmlFile = ideaDir.resolve(xmlConfigRelativePath)
                if (isXmlFileWithComponent(xmlFile)) return@run

                if (ideaDir.walk().filter { it.name.endsWith(".xml") && it != xmlFile }.any(::isXmlFileWithComponent)) return@run

                val document: Document
                val rootNode: Node
                if (xmlFile.isFile) {
                    document = parseDocument(xmlFile)
                    rootNode = document.documentElement
                } else {
                    document = newDocument()
                    rootNode = wrapperNodeCreator(document)
                }
                rootNode.appendIdeaComponentElement(componentName)
                xmlFile.outputStream().use { outputStream ->
                    createTransformer().withIdeaXmlConfigSettings().transform(document, outputStream)
                }
            }
        }

        run {
            IdeaConfigType.values().forEach {
                val xmlFile = rootProject.rootDir.resolve("${rootProject.name}.${it.name.toLowerCase()}")
                if (isXmlFileWithComponent(xmlFile)) return@run
            }
            val xmlFile = rootProject.rootDir.resolve("${rootProject.name}.${ideaConfigType.name.toLowerCase()}")
            if (xmlFile.isFile) {
                val document = parseDocument(xmlFile)
                document.documentElement.appendIdeaComponentElement(componentName)
                xmlFile.outputStream().use { outputStream ->
                    createTransformer().withIdeaXmlConfigSettings().transform(document, outputStream)
                }
            }
        }
    }
}


internal fun Document.hasIdeaComponentElement(componentName: String) = null != findIdeaComponentElement(componentName)

internal fun Document.findIdeaComponentElement(componentName: String): Element? = this.getElementsByTagName("component").firstOrNull {
    it is Element && componentName == it.getAttribute("name")
} as? Element

internal fun Node.appendIdeaComponentElement(componentName: String) = this.appendElement("component", mapOf("name" to componentName))

internal fun Node.addIdeaListStringValuesIfNotContains(values: Iterable<String>): Boolean {
    var wasChanged = false
    val list = this.findOrAppendElement("list") { wasChanged = true }
    val items = list.getChildElements("item").toMutableList()
    values.forEach { value ->
        if (items.none { value == it.getAttribute("itemvalue") }) {
            items += list.appendElement("item", mapOf(
                "class" to "java.lang.String",
                "itemvalue" to value
            ))
            wasChanged = true
        }
    }
    if (wasChanged) items.forEachIndexed { index, item -> item.setAttribute("index", index.toString()) }
    list.setAttribute("size", items.size.toString())
    return wasChanged
}


internal fun <T : Transformer> T.withIdeaXmlConfigSettings(): T = this.apply {
    setOutputProperty(OutputKeys.STANDALONE, "yes")
    setOutputProperty(OutputKeys.ENCODING, ENCODING)
    setOutputProperty(OutputKeys.METHOD, "xml")
    withIndention(2)
}
