package name.remal.gradle_plugins.utils

import name.remal.gradle_plugins.dsl.use
import org.w3c.dom.*
import java.io.*
import javax.xml.parsers.DocumentBuilder
import javax.xml.parsers.DocumentBuilderFactory
import javax.xml.transform.OutputKeys
import javax.xml.transform.Source
import javax.xml.transform.Transformer
import javax.xml.transform.TransformerFactory
import javax.xml.transform.dom.DOMResult
import javax.xml.transform.dom.DOMSource
import javax.xml.transform.stream.StreamResult
import javax.xml.transform.stream.StreamSource
import javax.xml.xpath.XPathConstants
import javax.xml.xpath.XPathExpression
import javax.xml.xpath.XPathFactory


private val documentBuilderFactory: DocumentBuilderFactory = DocumentBuilderFactory.newInstance().apply {
    isNamespaceAware = true
}
private val documentBuilder: DocumentBuilder = documentBuilderFactory.newDocumentBuilder()
internal fun newDocument(): Document = documentBuilder.newDocument()
internal fun parseDocument(file: File): Document = documentBuilder.parse(file)


private val transformerFactory = TransformerFactory.newInstance()!!
internal fun createTransformer(source: Source? = null, configurer: Transformer.() -> Unit = {}): Transformer {
    val transformer = if (null != source) transformerFactory.newTransformer(source) else transformerFactory.newTransformer()
    return transformer.apply(configurer)
}

internal fun createTransformer(reader: Reader, configurer: Transformer.() -> Unit = {}) = createTransformer(StreamSource(reader), configurer)
internal fun createTransformer(inputStream: InputStream, configurer: Transformer.() -> Unit = {}) = createTransformer(StreamSource(inputStream), configurer)
internal fun createTransformer(node: Node, configurer: Transformer.() -> Unit = {}) = createTransformer(DOMSource(node), configurer)

internal fun createTransformer(clazz: Class<*>, resourceName: String, configurer: Transformer.() -> Unit = {}) = createTransformer(clazz.getResourceAsStream(resourceName) ?: throw IllegalStateException("Resource can't be found: $resourceName for $clazz"), configurer)
internal fun createTransformer(classLoader: ClassLoader, resourceName: String, configurer: Transformer.() -> Unit = {}) = createTransformer(classLoader.getResourceAsStream(resourceName) ?: throw IllegalStateException("Resource can't be found: $resourceName"), configurer)

internal fun createStripSpaceTransformer(configurer: Transformer.() -> Unit = {}) = createTransformer(ClassLoaderProvider::class.java, "/strip-spaces.xslt", configurer)
private class ClassLoaderProvider


private val xpathFactory = XPathFactory.newInstance()!!
internal fun compileXPath(expression: String): XPathExpression = xpathFactory.newXPath().compile(expression)


// Node:

internal fun Node.remove(): Node = this.apply {
    this.parentNode.removeChild(this)
}

private val asStringTransformer = createTransformer()
private val asStringTransformerWithIndention = createTransformer().withIndention()

internal fun Node.asString(doIndent: Boolean = false): String {
    StringWriter().use { writer ->
        val transformer = if (!doIndent) asStringTransformer else asStringTransformerWithIndention
        transformer.transform(DOMSource(this), StreamResult(writer))
        return writer.toString()
    }
}

internal inline fun Node.getChildNodes(predicate: (Node) -> Boolean): List<Node> {
    val result = mutableListOf<Node>()
    this.childNodes.forEach {
        if (predicate(it)) {
            result.add(it)
        }
    }
    return result
}

internal fun Node.getChildElements(tagName: String): List<Element> {
    val result = mutableListOf<Element>()
    this.childNodes.forEach {
        if (it is Element) {
            if (it.tagName == tagName || it.localName == tagName) {
                result.add(it)
            }
        }
    }
    return result
}

internal fun Node.appendElement(tagName: String, attrs: Map<String, Any> = mapOf()): Element {
    val ownerDocument: Document = if (this is Document) this else this.ownerDocument
    val element = ownerDocument.createElement(tagName)
    attrs.forEach { name, value -> element.setAttribute(name, value.toString()) }
    this.appendChild(element)
    return element
}

internal fun Node.appendTextNode(data: String): Text {
    val ownerDocument: Document = if (this is Document) this else this.ownerDocument
    val textNode = ownerDocument.createTextNode(data)
    this.appendChild(textNode)
    return textNode
}

internal fun Node.findElement(tagName: String, attrs: Map<String, Any> = mapOf()): Element? {
    return this.getChildElements(tagName).firstOrNull { attrs.all { (name, value) -> it.getAttribute(name) == value.toString() } }
}

internal fun Node.findOrAppendElement(tagName: String, attrs: Map<String, Any> = mapOf(), onCreate: (Element) -> Unit = {}): Element {
    return this.findElement(tagName, attrs) ?: this.appendElement(tagName, attrs).also(onCreate)
}


internal operator fun Node.plusAssign(node: Node) {
    this.appendChild(node)
}

internal operator fun Node.plusAssign(text: String) {
    this.appendTextNode(text)
}


// NodeList:

internal val NodeList.isEmpty: Boolean get() = 0 == this.length
internal val NodeList.isNotEmpty: Boolean get() = 0 != this.length

internal inline fun NodeList.forEach(action: (Node) -> Unit) {
    if (isNotEmpty) {
        (0 until length).forEach { action(item(it)) }
    }
}

internal inline fun NodeList.firstOrNull(predicate: (Node) -> Boolean): Node? {
    this.forEach { if (predicate(it)) return it }
    return null;
}

internal inline fun NodeList.any(predicate: (Node) -> Boolean): Boolean {
    this.forEach { if (predicate(it)) return true }
    return false
}

internal inline fun NodeList.all(predicate: (Node) -> Boolean): Boolean {
    this.forEach { if (!predicate(it)) return false }
    return true
}

internal fun NodeList.toList(): List<Node> {
    val result = mutableListOf<Node>()
    this.forEach { result.add(it) }
    return result
}


// XPath:

internal fun XPathExpression.evaluateAsBoolean(item: Any): Boolean {
    return evaluate(item, XPathConstants.BOOLEAN) as Boolean
}

internal fun XPathExpression.evaluateAsString(item: Any): String? {
    return evaluate(item, XPathConstants.STRING) as? String
}

internal fun XPathExpression.evaluateAsNodeList(item: Any): NodeList {
    return evaluate(item, XPathConstants.NODESET) as NodeList
}


// Transformer:

internal const val DEFAULT_INDENT_SIZE = 4

internal fun Transformer.tryToSetOutputProperty(name: String, value: String?) {
    try {
        setOutputProperty(name, value)
    } catch (ignored: IllegalArgumentException) {
        /* unsupported by this transformer */
    }
}

internal fun <T : Transformer> T.withIndention(indentSize: Int = DEFAULT_INDENT_SIZE): T = this.apply {
    setOutputProperty(OutputKeys.INDENT, if (1 <= indentSize) "yes" else "no")

    if (1 <= indentSize) {
        arrayOf(
            "{http://xml.apache.org/xslt}indent-amount",
            "{http://saxon.sf.net/}indent-spaces"
        ).forEach { name ->
            tryToSetOutputProperty(name, indentSize.toString())
        }
    }
}

internal fun Transformer.transformToString(reader: Reader) = transformToString(StreamSource(reader))
internal fun Transformer.transformToString(inputStream: InputStream) = transformToString(StreamSource(inputStream))
internal fun Transformer.transformToString(node: Node) = transformToString(DOMSource(node))
internal fun Transformer.transformToString(source: Source): String {
    StringWriter().use { writer ->
        this.transform(source, StreamResult(writer))
        return writer.toString()
    }
}

internal fun Transformer.transformToDocument(reader: Reader) = transformToDocument(StreamSource(reader))
internal fun Transformer.transformToDocument(inputStream: InputStream) = transformToDocument(StreamSource(inputStream))
internal fun Transformer.transformToDocument(node: Node) = transformToDocument(DOMSource(node))
internal fun Transformer.transformToDocument(source: Source): Document {
    val document = newDocument()
    this.transform(source, DOMResult(document))
    return document
}

internal fun Transformer.transform(reader: Reader, writer: Writer) = transform(StreamSource(reader), writer)
internal fun Transformer.transform(inputStream: InputStream, writer: Writer) = transform(StreamSource(inputStream), writer)
internal fun Transformer.transform(node: Node, writer: Writer) = transform(DOMSource(node), writer)
internal fun Transformer.transform(source: Source, writer: Writer) {
    this.transform(source, StreamResult(writer))
}

internal fun Transformer.transform(reader: Reader, outputStream: OutputStream) = transform(StreamSource(reader), outputStream)
internal fun Transformer.transform(inputStream: InputStream, outputStream: OutputStream) = transform(StreamSource(inputStream), outputStream)
internal fun Transformer.transform(node: Node, outputStream: OutputStream) = transform(DOMSource(node), outputStream)
internal fun Transformer.transform(source: Source, outputStream: OutputStream) {
    this.transform(source, StreamResult(outputStream))
}
