package name.remal.building.gradle_plugins

import name.remal.building.gradle_plugins.artifact.ArtifactsCache
import name.remal.building.gradle_plugins.artifact.ArtifactsCacheCleanerPlugin
import name.remal.building.gradle_plugins.utils.*
import name.remal.building.gradle_plugins.utils.PluginIds.APPLICATION_PLUGIN_ID
import name.remal.building.gradle_plugins.utils.PluginIds.JAVA_PLUGIN_ID
import org.gradle.api.Project
import org.gradle.api.artifacts.Configuration
import org.gradle.api.distribution.DistributionContainer
import org.gradle.api.distribution.plugins.DistributionPlugin.MAIN_DISTRIBUTION_NAME
import org.gradle.api.file.FileCollection
import org.gradle.api.plugins.JavaPlugin.RUNTIME_CLASSPATH_CONFIGURATION_NAME
import org.gradle.api.tasks.application.CreateStartScripts
import org.gradle.api.tasks.testing.Test
import org.gradle.process.JavaExecSpec
import org.gradle.process.JavaForkOptions
import java.io.File
import java.nio.charset.Charset

open class AgentPlugin : ProjectPlugin() {

    companion object {
        const val AGENT_EXTENSION_NAME = "agentOptions"
        const val AGENT_CONFIGURATION_NAME = "agent"
        const val AGENT_AUTO_MANIFEST_ATTRIBUTE_NAME = "X-Agent-Auto"
        const val AGENT_ORDER_MANIFEST_ATTRIBUTE_NAME = "X-Agent-Order"
    }

    override fun apply(project: Project) {
        project.applyPlugin(ArtifactsCacheCleanerPlugin::class.java)

        project.withPlugin(JAVA_PLUGIN_ID) {

            val agentOptions = project.extensions.create(AGENT_EXTENSION_NAME, AgentExtension::class.java)

            val agentConf = project.configurations.create(AGENT_CONFIGURATION_NAME) {
                it.description = "Java agents"
                it.makeNotTransitive()
            }

            project.afterEvaluateOrdered(Int.MAX_VALUE) {
                project.tasks.configure { task ->
                    if (task !is JavaForkOptions) return@configure

                    val agentJars = getAgentJars(agentConf, run {
                        if (agentOptions.autoAppendAgentsFromClasspath) {
                            if (task is JavaExecSpec) {
                                task.classpath
                            } else if (task is Test) {
                                task.classpath
                            } else {
                                null
                            }
                        } else {
                            null
                        }
                    })
                    agentJars.forEach { task.jvmArgs("-javaagent:${it.path}") }
                }
            }

            project.withPlugin(APPLICATION_PLUGIN_ID) {
                project.afterEvaluateOrdered(Int.MAX_VALUE) {
                    val agentJars = getAgentJars(agentConf, run {
                        if (agentOptions.autoAppendAgentsFromClasspath) {
                            project.configurations[RUNTIME_CLASSPATH_CONFIGURATION_NAME]
                        } else {
                            null
                        }
                    })

                    val distribution = project[DistributionContainer::class.java][MAIN_DISTRIBUTION_NAME]
                    distribution.contents {
                        it.with(project.copySpec().apply {
                            into("lib")
                            agentJars.forEach { from(it) }
                        })
                    }

                    project.tasks.configure(CreateStartScripts::class.java) { task ->
                        task.doLastOrdered {
                            run {
                                var script = task.windowsScript.readText(Charset.defaultCharset())
                                if (!script.contains("\"%JAVA_EXE%\" %DEFAULT_JVM_OPTS%")) throw IllegalStateException("${task.windowsScript} do not contains '\"%JAVA_EXE%\" %DEFAULT_JVM_OPTS%'")
                                val agentsOpts = agentJars.map { "\"-javaagent:%APP_HOME%\\lib\\${it.name}\"" }.joinToString(" ")
                                script = script.replace("\"%JAVA_EXE%\" %DEFAULT_JVM_OPTS%", "\"%JAVA_EXE%\" $agentsOpts %DEFAULT_JVM_OPTS%")
                                task.windowsScript.writeText(script, Charset.defaultCharset())
                            }

                            run {
                                var script = task.unixScript.readText(Charset.defaultCharset())
                                if (!script.contains("exec \"\$JAVACMD\"")) throw IllegalStateException("${task.windowsScript} do not contains 'exec \"\$JAVACMD\"'")
                                val agentsOpts = agentJars.map { "\"-javaagent:\$APP_HOME/${it.name}\"" }.joinToString(" ")
                                script = script.replace("exec \"\$JAVACMD\"", "exec \"\$JAVACMD\" $agentsOpts")
                                task.unixScript.writeText(script, Charset.defaultCharset())
                            }
                        }
                    }
                }
            }

        }

    }

    private fun getAgentJars(files: FileCollection, additionalFiles: FileCollection? = null): List<File> {
        val agentFileOrders: MutableMap<File, Int> = LinkedHashMap()
        files.forEach { file ->
            agentFileOrders[file] = ArtifactsCache[file].manifestMainAttributes[AGENT_ORDER_MANIFEST_ATTRIBUTE_NAME]?.toIntOrNull() ?: 0
        }

        ((additionalFiles as? Configuration)?.copyRecursive() ?: additionalFiles)?.forEach { file ->
            val manifestMainAttributes = ArtifactsCache[file].manifestMainAttributes
            if (!(manifestMainAttributes[AGENT_AUTO_MANIFEST_ATTRIBUTE_NAME]?.toBoolean() ?: false)) return@forEach
            agentFileOrders[file] = manifestMainAttributes[AGENT_ORDER_MANIFEST_ATTRIBUTE_NAME]?.toIntOrNull() ?: 0
        }

        return agentFileOrders.keys.toList().sortedBy { agentFileOrders[it] }
    }

}


open class AgentExtension {
    var autoAppendAgentsFromClasspath: Boolean = true
}
