package name.remal.building.gradle_plugins

import name.remal.building.gradle_plugins.utils.ProjectPlugin
import org.gradle.api.Project
import org.gradle.api.ProjectConfigurationException
import org.jetbrains.annotations.NotNull

import static name.remal.building.gradle_plugins.PublishSettingsPlugin.MAVEN_PUBLICATION_NAME
import static name.remal.building.gradle_plugins.utils.PluginIds.BINTRAY_PLUGIN_ID
import static name.remal.building.gradle_plugins.utils.PluginIds.MAVEN_PUBLISH_PLUGIN_ID

class BintraySettingsPlugin extends ProjectPlugin {

    @Override
    void apply(@NotNull Project project) {
        project.pluginManager.withPlugin(BINTRAY_PLUGIN_ID) {
            project.pluginManager.apply(MAVEN_PUBLISH_PLUGIN_ID)
            project.pluginManager.apply(PublishSettingsPlugin.class)

            setupCredentials(project)
            setupPublications(project)
            setupBintrayUpload(project)
            setupPackage(project)
            setupVersion(project)
        }
    }

    private void setupCredentials(@NotNull Project project) {
        project.bintray {
            user = user ?: System.getenv('BINTRAY_USER')
            key = key ?: System.getenv('BINTRAY_API_KEY')
        }
    }

    private void setupPublications(@NotNull Project project) {
        project.bintray {
            publications = [MAVEN_PUBLICATION_NAME]
        }
    }

    private void setupBintrayUpload(@NotNull Project project) {
        project.tasks.getByName('bintrayUpload').doFirst { task ->
            if (null == task.user) task.logger.warn("Bintray user not set")
            if (null == task.apiKey) task.logger.warn("Bintray API key not set")

            if (task.API_URL_DEFAULT == task.apiUrl) {
                if (task.versionName?.toString()?.toLowerCase()?.contains("snapshot")) {
                    throw new ProjectConfigurationException("Snapshots can't be uploaded to ${task.apiUrl}")
                }
            }
        }
    }

    private void setupPackage(@NotNull Project project) {
        project.afterEvaluate {
            project.bintray.pkg {
                name = name ?: project.group
            }
        }
    }

    private void setupVersion(@NotNull Project project) {
        project.bintray.pkg.version {
            mavenCentralSync {
                user = user ?: System.getenv('OSS_USER')
                password = password ?: System.getenv('OSS_PASSWORD')
                close = '1'
            }
        }
    }

}
