package name.remal.building.gradle_plugins

import name.remal.building.gradle_plugins.utils.GenerateSourcesTask
import name.remal.building.gradle_plugins.utils.ProjectPlugin
import org.gradle.api.Project
import org.jetbrains.annotations.NotNull

import java.util.concurrent.atomic.AtomicBoolean

import static name.remal.building.gradle_plugins.JavaSettingsPluginKt.getGenerateJavaSourcesTaskName
import static name.remal.building.gradle_plugins.KotlinSettingsPluginKt.*
import static name.remal.building.gradle_plugins.utils.GradleUtilsKt.configure
import static name.remal.building.gradle_plugins.utils.JavaProjectKt.getJava
import static name.remal.building.gradle_plugins.utils.PluginIds.*

class GenerateSourcesPlugin extends ProjectPlugin {

    @Override
    void apply(@NotNull Project project) {
        project.pluginManager.withPlugin(JAVA_PLUGIN_ID) { setupJava(project) }
        project.pluginManager.withPlugin(KOTLIN_PLUGIN_ID) { setupKotlin(project) }
        project.pluginManager.withPlugin(KOTLIN2JS_PLUGIN_ID) { setupKotlin(project) }
    }

    private void setupJava(@NotNull Project project) {
        getJava(project).sourceSets.all { sourceSet ->
            GenerateSourcesTask generateSourcesTask = project.tasks.create(getGenerateJavaSourcesTaskName(sourceSet), GenerateSourcesTask.class) { task ->
                task.description = "Generates Java sources for ${sourceSet.name} source set"
                task.outputDir = new File(project.buildDir, "generated/java/${sourceSet.name}").absoluteFile
                task.classFileExtension = "java"
                sourceSet.java.srcDir(task.outputDir)
            }
            configure(project.tasks, getCompileKotlinTaskName(sourceSet)) { it.dependsOn(generateSourcesTask) }
            configure(project.tasks, getCompileKotlin2JSTaskName(sourceSet)) { it.dependsOn(generateSourcesTask) }
            configure(project.tasks, sourceSet.compileJavaTaskName) { it.dependsOn(generateSourcesTask) }
            project.afterEvaluate {
                project.tasks.findByName(sourceSet.compileJavaTaskName)?.with { task ->
                    generateSourcesTask.textFilesEncoding = task.options.encoding
                }
            }
        }
    }

    private final AtomicBoolean wasSetupKotlinExecuted = new AtomicBoolean(false)

    private void setupKotlin(@NotNull Project project) {
        if (!wasSetupKotlinExecuted.compareAndSet(false, true)) return

        getJava(project).sourceSets.all { sourceSet ->
            GenerateSourcesTask generateSourcesTask = project.tasks.create(getGenerateKotlinSourcesTaskName(sourceSet), GenerateSourcesTask.class) { task ->
                task.description = "Generates Kotlin sources for ${sourceSet.name} source set"
                task.outputDir = new File(project.buildDir, "generated/kotlin/${sourceSet.name}").absoluteFile
                task.classFileExtension = "kt"
                sourceSet.kotlin.srcDir(task.outputDir)
            }
            configure(project.tasks, getCompileKotlinTaskName(sourceSet)) { it.dependsOn(generateSourcesTask) }
            configure(project.tasks, getCompileKotlin2JSTaskName(sourceSet)) { it.dependsOn(generateSourcesTask) }
            project.afterEvaluate {
                project.tasks.findByName(sourceSet.compileJavaTaskName)?.with { task ->
                    generateSourcesTask.textFilesEncoding = task.options.encoding
                }
            }
        }
    }

}
