package name.remal.building.gradle_plugins

import name.remal.building.gradle_plugins.utils.PluginIds.JAVA_PLUGIN_ID
import name.remal.building.gradle_plugins.utils.ProjectPlugin
import name.remal.building.gradle_plugins.utils.get
import name.remal.building.gradle_plugins.utils.java
import name.remal.building.gradle_plugins.utils.withPlugin
import org.gradle.api.Project
import org.gradle.api.plugins.JavaPlugin.TEST_TASK_NAME
import org.gradle.api.tasks.SourceSet.MAIN_SOURCE_SET_NAME
import org.gradle.api.tasks.SourceSet.TEST_SOURCE_SET_NAME
import org.gradle.api.tasks.testing.Test

open class IntegrationTestingPlugin : ProjectPlugin() {

    companion object {
        const val INTEGRATION_SOURCE_SET_NAME = "integration"
        const val INTEGRATION_TEST_TASK_NAME = "integrationTest"
        const val ALL_TESTS_TASK_NAME = "allTests"
    }

    override fun apply(project: Project) {
        project.withPlugin(JAVA_PLUGIN_ID) {

            val integrationSourceSet = project.java.sourceSets.create(INTEGRATION_SOURCE_SET_NAME) { sourceSet ->
                sourceSet.java.srcDir(project.file("src/${sourceSet.name}/java"))
                sourceSet.resources.srcDir(project.file("src/${sourceSet.name}/resources"))
            }

            project.dependencies.apply {
                val mainSourceSet = project.java.sourceSets[MAIN_SOURCE_SET_NAME]
                val testSourceSet = project.java.sourceSets[TEST_SOURCE_SET_NAME]

                arrayOf(
                        project.configurations[testSourceSet.compileConfigurationName]
                        , mainSourceSet.output
                        , testSourceSet.output
                ).forEach {
                    add(integrationSourceSet.compileConfigurationName, it)
                }

                arrayOf(
                        project.configurations[testSourceSet.runtimeConfigurationName]
                ).forEach {
                    add(integrationSourceSet.runtimeConfigurationName, it)
                }
            }

            val integrationTestTask = project.tasks.create(INTEGRATION_TEST_TASK_NAME, Test::class.java) { task ->
                task.group = "Integration test"
                task.description = "Runs the integration tests"
                task.testClassesDirs = integrationSourceSet.output.classesDirs
                task.classpath = integrationSourceSet.runtimeClasspath
            }

            project.tasks.create(ALL_TESTS_TASK_NAME) { task ->
                task.group = integrationTestTask.group
                task.description = "Runs all tests"
                task.dependsOn(project.tasks[TEST_TASK_NAME])
                task.dependsOn(integrationTestTask)
            }

        }
    }

}