package name.remal.building.gradle_plugins

import mu.KLogging
import name.remal.building.gradle_plugins.artifact.ArtifactsCacheCleanerPlugin
import name.remal.building.gradle_plugins.artifact.CachedArtifactsCollection
import name.remal.building.gradle_plugins.utils.*
import name.remal.building.gradle_plugins.utils.Constants.*
import name.remal.building.gradle_plugins.utils.PluginIds.JAVA_PLUGIN_ID
import org.gradle.api.Project
import org.gradle.api.plugins.JavaPlugin.JAR_TASK_NAME
import org.gradle.api.plugins.JavaPlugin.TEST_TASK_NAME
import org.gradle.api.tasks.SourceSet
import org.gradle.api.tasks.compile.JavaCompile
import org.gradle.api.tasks.testing.Test
import org.gradle.api.tasks.testing.logging.TestExceptionFormat.FULL
import org.gradle.api.tasks.testing.logging.TestLogEvent
import org.gradle.api.tasks.testing.logging.TestLogEvent.FAILED
import org.gradle.api.tasks.testing.logging.TestLogEvent.STANDARD_ERROR
import org.gradle.api.tasks.testing.testng.TestNGOptions

open class JavaSettingsPlugin : ProjectPlugin() {

    companion object : KLogging() {
        const val COMPILE_ONLY_All_CONFIGURATION_NAME = "compileOnlyAll"
    }

    override fun apply(project: Project) {
        project.applyPlugin(ArtifactsCacheCleanerPlugin::class.java)

        project.withPlugin(JAVA_PLUGIN_ID) {
            setupRepositories(project)
            setupConfigurations(project)
            setupCompatibility(project)
            setupJavaCompile(project)
            setupTesting(project)
            setupCoverageMetrics(project)
        }
    }

    private fun setupRepositories(project: Project) {
        project.repositories.apply {
            if (isEmpty()) {
                mavenLocal()
                jcenter()
                mavenCentral()
            }
        }
    }

    private fun setupConfigurations(project: Project) {
        val compileOnlyAll = project.configurations.create(COMPILE_ONLY_All_CONFIGURATION_NAME) {
            it.description = "All compile-only configurations extends this configuration"
        }

        project.java.sourceSets.configure { sourceSet ->
            project.configurations.configure(sourceSet.compileOnlyConfigurationName) { configuration ->
                configuration.extendsFrom(compileOnlyAll)
            }
        }

        project.afterEvaluateOrdered {
            project.java.sourceSets.configure { sourceSet ->
                project.configurations.createIfNotExists(sourceSet.apiConfigurationName) { apiConf ->
                    project.configurations.findByName(sourceSet.compileConfigurationName)?.let { apiConf.extendsFrom(it) }
                    project.configurations.findByName(sourceSet.apiElementsConfigurationName)?.extendsFrom(apiConf)
                    project.configurations.findByName(sourceSet.implementationConfigurationName)?.extendsFrom(apiConf)
                }
            }
        }
    }

    private fun setupCompatibility(project: Project) {
        project.java.sourceCompatibility = DEFAULT_SOURCE_COMPATIBILITY
        project.java.targetCompatibility = DEFAULT_TARGET_COMPATIBILITY
    }

    private fun setupJavaCompile(project: Project) {
        project.configureTasks(JavaCompile::class.java) {
            it.options.apply {
                isDebug = true
                debugOptions.debugLevel = "source,lines,vars"
                isDeprecation = true
                encoding = encoding ?: ENCODING
            }
        }
    }

    private fun setupTesting(project: Project) {
        project.tasks[JAR_TASK_NAME].dependsOn(project.tasks[TEST_TASK_NAME])

        project.configureTasks(Test::class.java) { task ->
            task.enableAssertions = true

            task.testLogging.apply {
                showStackTraces = true
                setExceptionFormat(FULL as Any) // Gradle 3.5 compatibility
                events(FAILED, STANDARD_ERROR)
                info.events(*TestLogEvent.values())
            }
        }

        project.afterEvaluateOrdered {
            project.configureTasks(Test::class.java) { task ->
                if (task.isTestFrameworkSet) return@configureTasks

                val classpathArtifacts = CachedArtifactsCollection(task.classpath)
                if (classpathArtifacts.containsClass("org.testng.annotations.Test")) {
                    task.useTestNG()
                    task.maxParallelForks = 1
                    (task.options as TestNGOptions).apply {
                        parallel = "methods"
                        threadCount = Runtime.getRuntime().availableProcessors()
                    }

                } else if (classpathArtifacts.containsClass("org.junit.Test")) {
                    task.useJUnit()
                    task.maxParallelForks = Runtime.getRuntime().availableProcessors()
                }
            }
        }
    }

    private fun setupCoverageMetrics(project: Project) {
        project.applyPlugin(CoverageMetricsPlugin::class.java)
    }

}


val SourceSet.generateJavaSourcesTaskName: String get() = this.getTaskName("generate", "JavaSources")
