package name.remal.building.gradle_plugins

import com.google.common.collect.Multimap
import com.google.common.collect.MultimapBuilder
import mu.KLogging
import name.remal.building.gradle_plugins.utils.*
import name.remal.building.gradle_plugins.utils.PluginIds.JAVA_PLUGIN_ID
import org.gradle.api.Project
import org.gradle.api.file.CopySpec
import org.gradle.api.file.FileCollection
import org.gradle.api.internal.file.TemporaryFileProvider
import java.io.File
import java.io.IOException
import javax.inject.Inject

open class MergeJavaServicesPlugin @Inject constructor(
        private val temporaryFileProvider: TemporaryFileProvider
) : ProjectPlugin() {

    override fun apply(project: Project) {
        project.withPlugin(JAVA_PLUGIN_ID) {
            project.java.sourceSets.configure { sourceSet ->
                project.tasks.configure(sourceSet.jarTaskName) { task ->
                    task as CopySpec
                    task.doFirstOrdered {
                        val targetDir = temporaryFileProvider.newTemporaryFile("mergeJavaServices", sourceSet.name)
                        JavaServicesMerger(sourceSet.output, targetDir).merge()

                        task.from(targetDir)
                        task.exclude {
                            var parentFile = it.file.parentFile
                            if ("services" != parentFile.name) return@exclude false
                            parentFile = parentFile.parentFile
                            if ("META-INF" != parentFile.name) return@exclude false
                            parentFile = parentFile.parentFile
                            return@exclude parentFile != targetDir
                        }
                    }
                }
            }
        }
    }

}


private class JavaServicesMerger(val inputDirs: FileCollection, val targetDir: File) {

    private companion object : KLogging()

    fun merge(): Multimap<String, String> {
        logger.info("Merging Java services into {} from {}", targetDir, inputDirs.toList())

        if (!targetDir.deleteRecursively()) {
            throw IOException("$targetDir can't be deleted")
        }

        val services = MultimapBuilder.treeKeys().linkedHashSetValues().build<String, String>()
        inputDirs.forEach { inputDir ->
            inputDir.resolve(Constants.SERVICE_FILE_BASE_PATH).listFiles()?.filter(File::isFile)?.forEach { serviceFile ->
                serviceFile.forEachLine(Constants.TEXT_FILE_CHARSET) { line ->
                    line.substringBefore('#').trim().let { impl ->
                        services.put(serviceFile.name, impl)
                    }
                }
            }
        }

        if (!services.isEmpty) {
            for (serviceName in services.keySet()) {
                val targetFile = targetDir.resolve("${Constants.SERVICE_FILE_BASE_PATH}/$serviceName")
                targetFile.createParentDirectories().writer(Constants.TEXT_FILE_CHARSET).use { writer ->
                    services.get(serviceName).forEach {
                        writer.append(it).append("\n")
                    }
                }
            }
        }

        return services
    }

}
