package name.remal.building.gradle_plugins

import mu.KLogging
import name.remal.building.gradle_plugins.utils.*
import name.remal.building.gradle_plugins.utils.PluginIds.JAVA_PLUGIN_ID
import name.remal.building.gradle_plugins.utils.PluginIds.MAVEN_PUBLISH_PLUGIN_ID
import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.artifacts.Dependency.ARCHIVES_CONFIGURATION
import org.gradle.api.plugins.BasePlugin.BUILD_GROUP
import org.gradle.api.plugins.JavaBasePlugin.BUILD_TASK_NAME
import org.gradle.api.plugins.JavaPlugin.JAR_TASK_NAME
import org.gradle.api.publish.PublishingExtension
import org.gradle.api.publish.maven.MavenPublication
import org.gradle.api.publish.maven.plugins.MavenPublishPlugin.PUBLISH_LOCAL_LIFECYCLE_TASK_NAME
import org.gradle.api.tasks.SourceSet.MAIN_SOURCE_SET_NAME
import org.gradle.api.tasks.SourceSet.TEST_SOURCE_SET_NAME
import org.gradle.api.tasks.bundling.Jar
import org.gradle.internal.component.external.model.ModuleComponentArtifactIdentifier
import org.w3c.dom.Element

open class PublishSettingsPlugin : ProjectPlugin() {

    companion object : KLogging() {
        const val BUILD_INSTALL_TASK_NAME = "buildInstall"
        const val MAVEN_PUBLICATION_NAME = "mavenArchivesAuto"
    }

    override fun apply(project: Project) {
        project.withPlugin(JAVA_PLUGIN_ID) {
            val jar = project.tasks[JAR_TASK_NAME]

            val sourcesJar = project.tasks.create("sourcesJar", Jar::class.java) {
                it.classifier = "sources"
                it.from(project.java.sourceSets[MAIN_SOURCE_SET_NAME].allSource)
            }

            val javadocJar = project.tasks.create("emptyJavadocJar", Jar::class.java) {
                it.classifier = "javadoc"
            }

            setupArtifacts(project, jar, sourcesJar, javadocJar)

            project.withPlugin(MAVEN_PUBLISH_PLUGIN_ID) { setupMavenPublish(project) }
        }
    }

    private fun setupArtifacts(project: Project, vararg jarTasks: Task) {
        project.artifacts.apply {
            jarTasks.forEach { add(ARCHIVES_CONFIGURATION, it) }
        }
    }

    private fun setupMavenPublish(project: Project) {
        project.tasks.create(BUILD_INSTALL_TASK_NAME) {
            it.group = BUILD_GROUP
            it.dependsOn(project.tasks[BUILD_TASK_NAME])
            it.dependsOn(project.tasks[PUBLISH_LOCAL_LIFECYCLE_TASK_NAME])
        }

        project[PublishingExtension::class.java].apply {
            publications.configure(MavenPublication::class.java) { task ->
                task.pom.withXml { setupMavenPublishedPomXml(project, it.asElement()) }
            }
        }

        project[PublishingExtension::class.java].apply {
            publications.create(MAVEN_PUBLICATION_NAME, MavenPublication::class.java).apply {
                project.components.all { from(it) }
                project.configurations[ARCHIVES_CONFIGURATION].allArtifacts.all allArtifact@ { artifact ->
                    if (artifacts.any { it.extension.default() == artifact.extension.default() && it.classifier.default() == artifact.classifier.default() }) return@allArtifact
                    artifact(artifact)
                }
            }
        }
    }

    private fun setupMavenPublishedPomXml(project: Project, element: Element) {
        compileXPath("//dependencies/dependency[not(version) or not(version/text())]")
                .evaluateAsNodeList(element)
                .forEach forEachDependency@ { dependency ->
                    dependency as Element

                    val groupId = compileXPath("./groupId/text()").evaluateAsString(dependency)
                    val artifactId = compileXPath("./artifactId/text()").evaluateAsString(dependency)

                    val scope = compileXPath("./scope/text()").evaluateAsString(dependency)
                    val configuration = when (scope) {
                        "system" -> return@forEachDependency
                        "import" -> return@forEachDependency
                        "test" -> project.configurations[project.java.sourceSets[TEST_SOURCE_SET_NAME].apiConfigurationName]
                        "provided" -> project.configurations[project.java.sourceSets[MAIN_SOURCE_SET_NAME].compileOnlyConfigurationName]
                        "runtime" -> project.configurations[project.java.sourceSets[MAIN_SOURCE_SET_NAME].runtimeClasspathConfigurationName]
                        else -> project.configurations[project.java.sourceSets[MAIN_SOURCE_SET_NAME].apiConfigurationName]
                    }

                    configuration.incoming.artifacts.forEach forEachArtifact@ { artifact ->
                        val id = artifact.id as? ModuleComponentArtifactIdentifier ?: return@forEachArtifact
                        val componentId = id.componentIdentifier
                        if (componentId.group == groupId && componentId.module == artifactId) {
                            dependency.getElementsByTagName("version").forEach { it.parentNode.removeChild(it) }
                            dependency.appendChild(
                                    element.ownerDocument.createElement("version").apply {
                                        appendChild(
                                                element.ownerDocument.createTextNode(componentId.version)
                                        )
                                    }
                            )
                            return@forEachDependency
                        }
                    }
                }
    }

}