package name.remal.building.gradle_plugins.utils;

import org.gradle.api.DefaultTask;
import org.gradle.api.tasks.OutputDirectory;
import org.gradle.api.tasks.TaskAction;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.*;
import java.util.function.Consumer;

import static java.util.Objects.requireNonNull;
import static name.remal.building.gradle_plugins.utils.CommonKt.*;
import static name.remal.building.gradle_plugins.utils.Constants.ENCODING;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;
import static org.codehaus.groovy.runtime.ExceptionUtils.sneakyThrow;

public class GenerateSourcesTask extends DefaultTask {

    {
        getOutputs().doNotCacheIf("Task has custom actions", __ -> isHasCustomActions());
    }

    @OutputDirectory
    private File outputDir;

    public void setOutputDir(@NotNull File outputDir) {
        this.outputDir = requireNonNull(outputDir, "outputDir");
    }

    public File getOutputDir() {
        return outputDir;
    }

    @TaskAction
    protected void createOutputDirectory() {
        createDirectories(forceDeleteRecursively(getNotNullOutputDir()));
    }

    @NotNull
    protected File getNotNullOutputDir() {
        return requireNonNull(getOutputDir(), "outputDir");
    }

    private String classFileExtension;

    public void setClassFileExtension(@NotNull String classFileExtension) {
        this.classFileExtension = requireNonNull(classFileExtension, "classFileExtension");
    }

    public String getClassFileExtension() {
        return classFileExtension;
    }

    @NotNull
    protected String getNotNullClassFileExtension() {
        return requireNonNull(getClassFileExtension(), "classFileExtension");
    }

    private String textFilesEncoding;

    public void setTextFilesEncoding(@Nullable String textFilesEncoding) {
        this.textFilesEncoding = textFilesEncoding;
    }

    @NotNull
    public String getTextFilesEncoding() {
        String textFilesEncoding = this.textFilesEncoding;
        if (null == textFilesEncoding) textFilesEncoding = ENCODING;
        return textFilesEncoding;
    }

    public void addSourcesGeneration(@NotNull Consumer<File> action) {
        requireNonNull(action, "action");
        doLast(__ -> {
            action.accept(getNotNullOutputDir());
            setDidWork(true);
        });
    }

    public void addTextSourceGeneration(@NotNull String fileRelativePath, @NotNull Consumer<Writer> action) {
        requireNonNull(fileRelativePath, "fileRelativePath");
        requireNonNull(action, "action");
        addSourcesGeneration(outputDir -> {
            File classFile = new File(outputDir, fileRelativePath).getAbsoluteFile();
            getLogger().info("Generating {}", classFile);
            try (OutputStream outputStream = new FileOutputStream(createParentDirectories(classFile))) {
                try (Writer writer = new OutputStreamWriter(outputStream, getTextFilesEncoding())) {
                    action.accept(writer);
                }

            } catch (IOException e) {
                sneakyThrow(e);
            }
        });
    }

    public void addClassSourceGeneration(@Nullable String packageName, @NotNull String simpleClassName, @NotNull Consumer<Writer> action) {
        requireNonNull(simpleClassName, "simpleClassName");
        String fileRelativePath = simpleClassName.replace('.', '$') + '.' + getNotNullClassFileExtension();
        if (isNotEmpty(packageName)) fileRelativePath = packageName.replace('.', '/') + '/' + fileRelativePath;
        addTextSourceGeneration(fileRelativePath, action);
    }

}
