package name.remal.building.gradle_plugins.utils

import name.remal.building.gradle_plugins.utils.Constants.CLASS_FILE_NAME_SUFFIX
import name.remal.building.gradle_plugins.utils.Constants.ENCODING
import java.io.File
import java.io.IOException
import java.lang.reflect.Field
import java.net.URLDecoder
import java.net.URLEncoder
import java.nio.file.FileVisitResult
import java.nio.file.Files
import java.nio.file.Path
import java.nio.file.SimpleFileVisitor
import java.nio.file.attribute.BasicFileAttributes
import java.time.Duration
import java.util.Collections.newSetFromMap
import java.util.concurrent.ConcurrentHashMap
import java.util.stream.Stream
import java.util.stream.StreamSupport.stream
import java.util.zip.ZipException
import java.util.zip.ZipFile

val DEFAULT_IO_TIMEOUT = Duration.ofSeconds(5)!!

fun classToResourcePath(className: String) = className.replace('.', '/') + CLASS_FILE_NAME_SUFFIX
fun classToResourcePath(clazz: Class<*>) = classToResourcePath(clazz.name)

fun resourcePathToClassName(path: String) = path.replace('/', '.').replace('\\', '.').trim('.').run { substring(0, length - CLASS_FILE_NAME_SUFFIX.length) }

fun isZipFile(file: File): Boolean {
    if (file.isFile) {
        try {
            ZipFile(file).use {
                it.entries()
                return true
            }
        } catch (e: ZipException) {
            return false
        }
    }
    return false
}

inline fun forEachClassFile(classedDir: File, action: (File) -> Unit) {
    classedDir.walk()
            .filter { it.name.endsWith(CLASS_FILE_NAME_SUFFIX) }
            .filter(File::isFile)
            .forEach(action)
}

fun <T> concurrentSetOf(): MutableSet<T> {
    return newSetFromMap<T>(ConcurrentHashMap())
}

fun <T> concurrentSetOf(vararg elements: T): MutableSet<T> {
    return newSetFromMap<T>(ConcurrentHashMap()).apply {
        addAll(elements)
    }
}


fun <C : Any, O : C> Class<C>.getField(obj: O, name: String): Any? {
    return getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.get(obj)
}

fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Any?) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.set(obj, value)
}

fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Boolean) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setBoolean(obj, value)
}

fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Byte) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setByte(obj, value)
}

fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Char) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setChar(obj, value)
}

fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Short) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setShort(obj, value)
}

fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Int) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setInt(obj, value)
}

fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Long) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setLong(obj, value)
}

fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Float) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setFloat(obj, value)
}

fun <C : Any, O : C> Class<C>.setField(obj: O, name: String, value: Double) {
    getDeclaredFieldFromHierarchy(name).apply { isAccessible = true }.setDouble(obj, value)
}

fun Class<*>.getDeclaredFieldFromHierarchy(name: String): Field {
    var currentClass = this
    while (true) {
        try {
            return currentClass.getDeclaredField(name)
        } catch (e: NoSuchFieldException) {
            if (Any::class.java == currentClass) throw e
            currentClass = currentClass.superclass
        }
    }
}

val Class<*>.isConcreteClass: Boolean get() = !isInterface && !isEnum && !isPrimitive && isArray && !isAnonymousClass && !isLocalClass && !isSynthetic

fun String?.default(value: String = ""): String = this ?: value

fun String?.nullIfEmpty(): String? = if (null == this || this.isEmpty()) null else this

fun String.encodeURIComponent(): String = URLEncoder.encode(this, ENCODING)
fun String.decodeURIComponent(): String = URLDecoder.decode(this, ENCODING)

@Suppress("DEPRECATION")
fun String.escapeJava(): String = org.apache.commons.lang3.StringEscapeUtils.escapeJava(this)

fun StringBuilder.clear(): StringBuilder = this.apply { setLength(0) }

fun File.forceDelete(): File {
    if (this.exists()) Files.delete(this.toPath())
    return this
}

fun File.forceDeleteRecursively(): File {
    if (this.exists()) Files.walkFileTree(this.toPath(), object : SimpleFileVisitor<Path>() {
        override fun visitFile(file: Path, attrs: BasicFileAttributes): FileVisitResult {
            Files.delete(file)
            return FileVisitResult.CONTINUE
        }

        override fun postVisitDirectory(dir: Path, exc: IOException?): FileVisitResult {
            Files.delete(dir)
            return FileVisitResult.CONTINUE
        }
    })
    return this
}

fun File.createDirectories(): File {
    Files.createDirectories(this.absoluteFile.toPath())
    return this
}

fun File.createParentDirectories(): File {
    Files.createDirectories(this.absoluteFile.parentFile.toPath())
    return this
}

fun <T, R : T> Stream<T>.filterIsInstance(type: Class<R>): Stream<R> {
    return this.filter(type::isInstance).map(type::cast)
}

@JvmName("flatMapCollection")
fun <T, R> Stream<T>.flatMap(mapper: (T) -> Collection<R>): Stream<R> {
    return this.flatMap { mapper(it).stream() }
}

@JvmName("flatMapIterable")
fun <T, R> Stream<T>.flatMap(mapper: (T) -> Iterable<R>): Stream<R> {
    return this.flatMap { stream(mapper(it).spliterator(), false) }
}

inline fun <T : AutoCloseable?, R> T.use(block: (T) -> R): R {
    var closed = false
    try {
        return block(this)

    } catch (e: Exception) {
        closed = true
        try {
            this?.close()
        } catch (closeException: Exception) {
            e.addSuppressed(closeException)
        }
        throw e

    } finally {
        if (!closed) {
            this?.close()
        }
    }
}
