package name.remal.building.gradle_plugins.utils

import org.gradle.api.*
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ConfigurationContainer
import org.gradle.api.artifacts.ExcludeRule.GROUP_KEY
import org.gradle.api.artifacts.ExcludeRule.MODULE_KEY
import org.gradle.api.artifacts.ModuleDependency
import org.gradle.api.artifacts.result.ResolvedDependencyResult
import org.gradle.api.attributes.Attribute
import org.gradle.api.internal.artifacts.configurations.ConfigurationInternal
import org.gradle.api.internal.project.ProjectStateInternal
import org.gradle.api.plugins.AppliedPlugin
import org.gradle.api.plugins.Convention
import org.gradle.api.plugins.ExtensionContainer
import org.gradle.api.plugins.UnknownPluginException
import org.gradle.api.tasks.TaskCollection
import org.gradle.api.tasks.TaskState
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicInteger

// Project:

val Project.isRootProject: Boolean get() = null == this.parent

val Project.isEvaluating: Boolean get() = !this.isExecuting && !this.isExecuted && !this.hasFailure
val Project.isExecuting: Boolean get() = (this.state as ProjectStateInternal).executing
val Project.isExecuted: Boolean get() = this.state.executed
val Project.hasFailure: Boolean get() = null != this.state.failure

operator fun <T : Any> Project.get(type: Class<T>): T {
    try {
        return this.convention[type]

    } catch (conventionException: Throwable) {
        try {
            return this.extensions[type]

        } catch (e: Throwable) {
            e.addSuppressed(conventionException)
            throw e
        }
    }
}

fun Project.applyPlugin(id: String, action: () -> Unit = {}) {
    this.pluginManager.apply(id)
    action()
}

fun Project.applyPluginIfExists(id: String, action: () -> Unit = {}) {
    try {
        this.pluginManager.apply(id);
    } catch(e: UnknownPluginException) {
        return // do nothing
    }

    action()
}

fun <T : Plugin<Project>> Project.applyPlugin(type: Class<T>, action: () -> Unit = {}) {
    this.pluginManager.apply(type)
    action()
}

fun <T : Task> Project.configureTasks(type: Class<T>, configurer: (T) -> Unit) {
    this.tasks.configure(type, configurer)
}

fun Project.withPlugin(id: String, action: (Project) -> Unit) {
    val project = this
    this.pluginManager.withPlugin(id, { action(project) })
}

fun Project.withPlugins(vararg ids: String, action: (Project) -> Unit) {
    if (ids.isEmpty()) throw IllegalArgumentException("empty ids")
    val project = this
    val counter = AtomicInteger(ids.size)
    val callback = Action<AppliedPlugin> {
        if (0 == counter.decrementAndGet()) {
            action(project)
        }
    }
    for (id in ids) {
        this.pluginManager.withPlugin(id, callback)
    }
}

fun Project.withOneOfPlugin(vararg ids: String, action: (Project) -> Unit) {
    if (ids.isEmpty()) throw IllegalArgumentException("empty ids")
    val project = this
    val isExecuted = AtomicBoolean(false)
    val callback = Action<AppliedPlugin> {
        if (isExecuted.compareAndSet(false, true)) {
            action(project)
        }
    }
    for (id in ids) {
        this.pluginManager.withPlugin(id, callback)
    }
}

fun Project.afterEvaluateOrNow(action: (Project) -> Unit) {
    if (this.isEvaluating) {
        this.afterEvaluate(action)
    } else {
        this.state.rethrowFailure()
        action(this)
    }
}

fun Project.afterEvaluate(action: ProjectAction) {
    val ext = this.extensions.getOrCreate(
            AfterEvaluateProjectActionsExtension::class.java,
            onCreate = { ext ->
                this.afterEvaluate {
                    ext.actions.sorted().forEach { it.execute(this) }
                }
            }
    )
    ext.actions.add(action)
}

fun Project.afterEvaluateOrdered(order: Int = 0, action: (Project) -> Unit) {
    afterEvaluate(object : ProjectAction {
        override fun execute(project: Project) {
            action(project)
        }

        override fun getOrder(): Int {
            return order
        }
    })
}


// Task:

@Suppress("UNCHECKED_CAST")
fun <TaskType : Task> TaskType.doFirst(action: TaskAction<TaskType>) {
    val ext = this.extensions.getOrCreate(
            DoFirstTaskActionsExtension::class.java,
            onCreate = { ext ->
                ext as DoFirstTaskActionsExtension<TaskType>
                this.doFirst {
                    ext.actions.sorted().forEach { it.execute(this) }
                }
            }
    )
    ext as DoFirstTaskActionsExtension<TaskType>
    ext.actions.add(action)
}

fun <TaskType : Task> TaskType.doFirstOrdered(order: Int = 0, action: (TaskType) -> Unit) {
    doFirst(object : TaskAction<TaskType> {
        override fun execute(task: TaskType) {
            action(task)
        }

        override fun getOrder(): Int {
            return order
        }
    })
}

@Suppress("UNCHECKED_CAST")
fun <TaskType : Task> TaskType.doLast(action: TaskAction<TaskType>) {
    val ext = this.extensions.getOrCreate(
            DoLastTaskActionsExtension::class.java,
            onCreate = { ext ->
                ext as DoLastTaskActionsExtension<TaskType>
                this.doLast {
                    ext.actions.sorted().forEach { it.execute(this) }
                }
            }
    )
    ext as DoLastTaskActionsExtension<TaskType>
    ext.actions.add(action)
}

fun <TaskType : Task> TaskType.doLastOrdered(order: Int = 0, action: (TaskType) -> Unit) {
    doLast(object : TaskAction<TaskType> {
        override fun execute(task: TaskType) {
            action(task)
        }

        override fun getOrder(): Int {
            return order
        }
    })
}


// TaskState:

val TaskState.failed: Boolean get() = null != this.failure


// ConfigurationContainer:

fun ConfigurationContainer.findRootConfigurations(): NamedDomainObjectSet<Configuration> {
    return this.matching { it.extendsFrom.isEmpty() }
}

fun ConfigurationContainer.resolveCopyOf(vararg configurations: Configuration) = resolveCopyOf(configurations.toList())

fun ConfigurationContainer.resolveCopyOf(configurations: Collection<Configuration>): Set<ResolvedDependencyResult> {
    if (configurations.isEmpty()) return setOf()

    fun Configuration.extendsFromCopy(conf: Configuration) {
        extendsFrom(detachedConfiguration().apply {
            attributes.let { attrs ->
                conf.attributes.let { confAttrs ->
                    confAttrs.keySet().forEach { key ->
                        val value = confAttrs.getAttribute(key) as Any?
                        @Suppress("UNCHECKED_CAST")
                        attrs.attribute(key as Attribute<Any?>, value)
                        Unit
                    }
                }
            }
            javaClass.setField(this, "resolutionStrategy", resolutionStrategy)
            isTransitive = conf.isTransitive
            dependencies.addAll(conf.dependencies)
            conf.excludeRules.forEach {
                val excludeProps = mutableMapOf<String, String?>()
                it.group?.let { excludeProps[GROUP_KEY] = it }
                it.module?.let { excludeProps[MODULE_KEY] = it }
                exclude(excludeProps)
            }
            javaClass.setField(this, "defaultDependencyActions", conf.javaClass.getField(conf, "defaultDependencyActions"))
            conf.extendsFrom.forEach { extendsFromCopy(it) }
        })
    }

    val rootConf = this.detachedConfiguration()
    configurations.forEach { rootConf.extendsFromCopy(it) }

    rootConf as ConfigurationInternal
    rootConf.triggerWhenEmptyActionsIfNecessary()

    if (rootConf.allDependencies.isEmpty()) return emptySet()

    return rootConf.incoming.resolutionResult.allDependencies.asSequence()
            .filterIsInstance(ResolvedDependencyResult::class.java)
            .toSet()
}


// Configuration:

fun Configuration.makeNotTransitive() {
    this.isTransitive = false
    this.dependencies.configure { dep ->
        if (dep is ModuleDependency) dep.isTransitive = false
    }
}


// ExtensionContainer:

operator fun <T : Any> ExtensionContainer.get(type: Class<T>): T {
    var supressedException: Throwable? = null
    try {
        if (this is Convention) {
            return this.getPlugin(type)
        }
    } catch (e: Throwable) {
        supressedException = e
    }

    try {
        return this.getByType(type)
    } catch (e: Throwable) {
        e.addSuppressed(supressedException)
        throw e
    }
}

fun <Type : Any, ImplType : Type> ExtensionContainer.getOrCreate(type: Class<Type>, name: String? = null, implType: Class<ImplType>? = null, vararg arguments: Any?, onCreate: (Type) -> Unit = {}): Type {
    this.findByType(type)?.let { return it }
    return this.create(type, name ?: type.name.replace('.', '_'), implType ?: type, *arguments).apply {
        onCreate(this)
    }
}


// DomainObjectCollection:

operator fun <T, R : T> DomainObjectCollection<T>.get(type: Class<R>): DomainObjectCollection<R> {
    return this.withType(type)
}

fun <T> DomainObjectCollection<T>.configure(configurer: (T) -> Unit) {
    this.all(configurer)
}

fun <T, R : T> DomainObjectCollection<T>.configure(type: Class<R>, configurer: (R) -> Unit) {
    this[type].all(configurer)
}


// NamedDomainObjectCollection:

operator fun <T> NamedDomainObjectCollection<T>.get(name: String): T {
    return this.getByName(name)
}

fun <T> NamedDomainObjectContainer<T>.getOrCreate(name: String, configurer: (T) -> Unit = {}): T {
    return findByName(name) ?: create(name, configurer)
}

fun <T> NamedDomainObjectContainer<T>.createIfNotExists(name: String, configurer: (T) -> Unit = {}) {
    getOrCreate(name, configurer)
}

fun <T> NamedDomainObjectCollection<T>.configure(name: String, configurer: (T) -> Unit) {
    val hasObjectWithName = AtomicBoolean(false)
    this.all {
        this.findByName(name)?.let {
            if (hasObjectWithName.compareAndSet(false, true)) {
                configurer(it)
            }
        }
    }
    this.whenObjectRemoved {
        if (null == this.findByName(name)) {
            hasObjectWithName.set(false)
        }
    }
}

operator fun <T, R : T> NamedDomainObjectCollection<T>.get(type: Class<R>): NamedDomainObjectCollection<R> {
    return this.withType(type)
}

// other:

operator fun <T : Task, R : T> TaskCollection<T>.get(type: Class<R>): TaskCollection<R> {
    return this.withType(type)
}

