package name.remal.building.gradle_plugins.utils

import name.remal.building.gradle_plugins.utils.Constants.ENCODING
import org.gradle.plugins.ide.idea.model.IdeaModel
import org.w3c.dom.Document
import org.w3c.dom.Element
import org.w3c.dom.Node
import java.io.File
import javax.xml.transform.OutputKeys
import javax.xml.transform.Transformer


fun IdeaModel.setup(action: (IdeaModel) -> Unit) {
    val ideaModel = this
    this.project.project.gradle.buildFinished {
        action(ideaModel)
    }
}

fun IdeaModel.setup(vararg actions: (IdeaModel) -> Unit) {
    val ideaModel = this
    this.project.project.gradle.buildFinished {
        actions.forEach { it(ideaModel) }
    }
}


fun IdeaModel.processConfigs(action: (Document) -> Document?) {
    this.project.project.rootProject.let { rootProject ->
        fun processXmlFile(xmlFile: File) {
            val source = parseDocument(xmlFile)
            val result = action(source)
            if (null != result) {
                xmlFile.outputStream().use { outputStream ->
                    createStripSpaceTransformer().withIdeaXmlConfigSettings().transform(result, outputStream)
                }
            }
        }

        rootProject.rootDir.resolve(".idea").walk()
                .filter(File::isFile)
                .filter { it.name.endsWith(".xml") }
                .forEach(::processXmlFile)

        arrayOf("ipr", "iws").forEach { ext ->
            rootProject.rootDir.resolve("${rootProject.name}.$ext").let {
                if (it.isFile) processXmlFile(it)
            }
        }
    }
}


enum class IdeaConfigType {
    IPR,
    IWS,
}

val IDEA_CONFIG_PROJECT_VERSION = "4"

fun IdeaModel.writeComponentIfNotExists(componentName: String, ideaConfigType: IdeaConfigType, xmlConfigRelativePath: String, wrapperNodeCreator: (Document) -> Node = { it }) {
    this.project.project.rootProject.let { rootProject ->
        fun isXmlFileWithComponent(xmlFile: File): Boolean {
            if (!xmlFile.isFile) return false
            val document = parseDocument(xmlFile)
            return document.hasIdeaComponentElement(componentName)
        }

        run {
            val ideaDir = rootProject.rootDir.resolve(".idea")
            if (ideaDir.isDirectory) {
                val xmlFile = ideaDir.resolve(xmlConfigRelativePath)
                if (isXmlFileWithComponent(xmlFile)) return@run

                if (ideaDir.walk().filter { it.name.endsWith(".xml") && it != xmlFile }.any(::isXmlFileWithComponent)) return@run

                val document: Document
                val rootNode: Node
                if (xmlFile.isFile) {
                    document = parseDocument(xmlFile)
                    rootNode = document.documentElement
                } else {
                    document = newDocument()
                    rootNode = wrapperNodeCreator(document)
                }
                rootNode.appendIdeaComponentElement(componentName)
                xmlFile.outputStream().use { outputStream ->
                    createTransformer().withIdeaXmlConfigSettings().transform(document, outputStream)
                }
            }
        }

        run {
            IdeaConfigType.values().forEach {
                val xmlFile = rootProject.rootDir.resolve("${rootProject.name}.${it.name.toLowerCase()}")
                if (isXmlFileWithComponent(xmlFile)) return@run
            }
            val xmlFile = rootProject.rootDir.resolve("${rootProject.name}.${ideaConfigType.name.toLowerCase()}")
            if (xmlFile.isFile) {
                val document = parseDocument(xmlFile)
                document.documentElement.appendIdeaComponentElement(componentName)
                xmlFile.outputStream().use { outputStream ->
                    createTransformer().withIdeaXmlConfigSettings().transform(document, outputStream)
                }
            }
        }
    }
}


fun Document.hasIdeaComponentElement(componentName: String) = null != findIdeaComponentElement(componentName)

fun Document.findIdeaComponentElement(componentName: String): Element? = this.getElementsByTagName("component").firstOrNull {
    it is Element && componentName == it.getAttribute("name")
} as? Element

fun Node.appendIdeaComponentElement(componentName: String) = this.appendElement("component", mapOf("name" to componentName))

fun Node.addIdeaListStringValuesIfNotContains(values: Iterable<String>): Boolean {
    var wasChanged = false
    val list = this.findOrAppendElement("list") { wasChanged = true }
    val items = list.getChildElements("item").toMutableList()
    values.forEach { value ->
        if (items.none { value == it.getAttribute("itemvalue") }) {
            items += list.appendElement("item", mapOf(
                    "class" to "java.lang.String",
                    "itemvalue" to value
            ))
            wasChanged = true
        }
    }
    if (wasChanged) items.forEachIndexed { index, item -> item.setAttribute("index", index.toString()) }
    list.setAttribute("size", items.size.toString())
    return wasChanged
}


fun <T : Transformer> T.withIdeaXmlConfigSettings(): T = this.apply {
    setOutputProperty(OutputKeys.STANDALONE, "yes")
    setOutputProperty(OutputKeys.ENCODING, ENCODING)
    setOutputProperty(OutputKeys.METHOD, "xml")
    withIndention(2)
}
