package name.remal.building.gradle_plugins.utils

import org.w3c.dom.*
import java.io.*
import javax.xml.parsers.DocumentBuilder
import javax.xml.parsers.DocumentBuilderFactory
import javax.xml.transform.OutputKeys
import javax.xml.transform.Source
import javax.xml.transform.Transformer
import javax.xml.transform.TransformerFactory
import javax.xml.transform.dom.DOMResult
import javax.xml.transform.dom.DOMSource
import javax.xml.transform.stream.StreamResult
import javax.xml.transform.stream.StreamSource
import javax.xml.xpath.XPathConstants
import javax.xml.xpath.XPathExpression
import javax.xml.xpath.XPathFactory


private val documentBuilderFactory: DocumentBuilderFactory = DocumentBuilderFactory.newInstance().apply {
    isNamespaceAware = true
}
private val documentBuilder: DocumentBuilder = documentBuilderFactory.newDocumentBuilder()
fun newDocument(): Document = documentBuilder.newDocument()
fun parseDocument(file: File): Document = documentBuilder.parse(file)


private val transformerFactory = TransformerFactory.newInstance()!!
fun createTransformer(source: Source? = null, configurer: Transformer.() -> Unit = {}): Transformer {
    val transformer = if (null != source) transformerFactory.newTransformer(source) else transformerFactory.newTransformer()
    return transformer.apply(configurer)
}

fun createTransformer(reader: Reader, configurer: Transformer.() -> Unit = {}) = createTransformer(StreamSource(reader), configurer)
fun createTransformer(inputStream: InputStream, configurer: Transformer.() -> Unit = {}) = createTransformer(StreamSource(inputStream), configurer)
fun createTransformer(node: Node, configurer: Transformer.() -> Unit = {}) = createTransformer(DOMSource(node), configurer)

fun createTransformer(clazz: Class<*>, resourceName: String, configurer: Transformer.() -> Unit = {}) = createTransformer(clazz.getResourceAsStream(resourceName) ?: throw IllegalStateException("Resource can't be found: $resourceName for $clazz"), configurer)
fun createTransformer(classLoader: ClassLoader, resourceName: String, configurer: Transformer.() -> Unit = {}) = createTransformer(classLoader.getResourceAsStream(resourceName) ?: throw IllegalStateException("Resource can't be found: $resourceName"), configurer)

fun createStripSpaceTransformer(configurer: Transformer.() -> Unit = {}) = createTransformer(ClassLoaderProvider::class.java, "/strip-spaces.xslt", configurer)
private class ClassLoaderProvider


private val xpathFactory = XPathFactory.newInstance()!!
fun compileXPath(expression: String): XPathExpression = xpathFactory.newXPath().compile(expression)


// Node:

fun Node.remove(): Node = this.apply {
    this.parentNode.removeChild(this)
}

fun Node.asString(doIndent: Boolean = false): String {
    return StringWriter().use { writer ->
        val transformer = createTransformer()
        if (doIndent) transformer.withIndention()
        transformer.transform(DOMSource(this), StreamResult(writer))
    }.toString()
}

inline fun Node.getChildNodes(predicate: (Node) -> Boolean): List<Node> {
    val result = mutableListOf<Node>()
    this.childNodes.forEach {
        if (predicate(it)) {
            result.add(it)
        }
    }
    return result
}

fun Node.getChildElements(tagName: String): List<Element> {
    val result = mutableListOf<Element>()
    this.childNodes.forEach {
        if (it is Element && it.localName == tagName) {
            result.add(it)
        }
    }
    return result
}

fun Node.appendElement(tagName: String, attrs: Map<String, Any> = mapOf()): Element {
    val ownerDocument: Document = if (this is Document) this else this.ownerDocument
    val element = ownerDocument.createElement(tagName)
    attrs.forEach { name, value -> element.setAttribute(name, value.toString()) }
    this.appendChild(element)
    return element
}

fun Node.appendTextNode(data: String): Text {
    val ownerDocument: Document = if (this is Document) this else this.ownerDocument
    val textNode = ownerDocument.createTextNode(data)
    this.appendChild(textNode)
    return textNode
}

fun Node.findElement(tagName: String, attrs: Map<String, Any> = mapOf()): Element? {
    return this.getChildElements(tagName)
            .filter { attrs.all { (name, value) -> it.getAttribute(name) == value.toString() } }
            .firstOrNull()
}

fun Node.findOrAppendElement(tagName: String, attrs: Map<String, Any> = mapOf(), onCreate: (Element) -> Unit = {}): Element {
    return this.findElement(tagName, attrs) ?: this.appendElement(tagName, attrs).also(onCreate)
}


// NodeList:

val NodeList.isEmpty: Boolean get() = 0 == this.length
val NodeList.isNotEmpty: Boolean get() = 0 != this.length

inline fun NodeList.forEach(action: (Node) -> Unit) {
    if (isNotEmpty) {
        (0..length - 1).forEach { action(item(it)) }
    }
}

inline fun NodeList.firstOrNull(predicate: (Node) -> Boolean): Node? {
    this.forEach { if (predicate(it)) return it }
    return null;
}

inline fun NodeList.any(predicate: (Node) -> Boolean): Boolean {
    this.forEach { if (predicate(it)) return true }
    return false
}

inline fun NodeList.all(predicate: (Node) -> Boolean): Boolean {
    this.forEach { if (!predicate(it)) return false }
    return true
}

fun NodeList.toList(): List<Node> {
    val result = mutableListOf<Node>()
    this.forEach { result.add(it) }
    return result
}


// XPath:

fun XPathExpression.evaluateAsBoolean(item: Any): Boolean {
    return evaluate(item, XPathConstants.BOOLEAN) as Boolean
}

fun XPathExpression.evaluateAsString(item: Any): String? {
    return evaluate(item, XPathConstants.STRING) as? String
}

fun XPathExpression.evaluateAsNodeList(item: Any): NodeList {
    return evaluate(item, XPathConstants.NODESET) as NodeList
}


// Transformer:

const val DEFAULT_INDENT_SIZE = 4

fun Transformer.tryToSetOutputProperty(name: String, value: String?) {
    try {
        setOutputProperty(name, value)
    } catch (ignored: IllegalArgumentException) {
        /* unsupported by transformer */
    }
}

fun <T : Transformer> T.withIndention(indentSize: Int = DEFAULT_INDENT_SIZE): T = this.apply {
    setOutputProperty(OutputKeys.INDENT, if (1 <= indentSize) "yes" else "no")

    if (1 <= indentSize) {
        arrayOf(
                "{http://xml.apache.org/xslt}indent-amount",
                "{http://saxon.sf.net/}indent-spaces"
        ).forEach { name ->
            tryToSetOutputProperty(name, indentSize.toString())
        }
    }
}

fun Transformer.transformToString(reader: Reader) = transformToString(StreamSource(reader))
fun Transformer.transformToString(inputStream: InputStream) = transformToString(StreamSource(inputStream))
fun Transformer.transformToString(node: Node) = transformToString(DOMSource(node))
fun Transformer.transformToString(source: Source): String {
    StringWriter().use { writer ->
        this.transform(source, StreamResult(writer))
        return writer.toString()
    }
}

fun Transformer.transformToDocument(reader: Reader) = transformToDocument(StreamSource(reader))
fun Transformer.transformToDocument(inputStream: InputStream) = transformToDocument(StreamSource(inputStream))
fun Transformer.transformToDocument(node: Node) = transformToDocument(DOMSource(node))
fun Transformer.transformToDocument(source: Source): Document {
    val document = newDocument()
    this.transform(source, DOMResult(document))
    return document
}

fun Transformer.transform(reader: Reader, writer: Writer) = transform(StreamSource(reader), writer)
fun Transformer.transform(inputStream: InputStream, writer: Writer) = transform(StreamSource(inputStream), writer)
fun Transformer.transform(node: Node, writer: Writer) = transform(DOMSource(node), writer)
fun Transformer.transform(source: Source, writer: Writer) {
    this.transform(source, StreamResult(writer))
}

fun Transformer.transform(reader: Reader, outputStream: OutputStream) = transform(StreamSource(reader), outputStream)
fun Transformer.transform(inputStream: InputStream, outputStream: OutputStream) = transform(StreamSource(inputStream), outputStream)
fun Transformer.transform(node: Node, outputStream: OutputStream) = transform(DOMSource(node), outputStream)
fun Transformer.transform(source: Source, outputStream: OutputStream) {
    this.transform(source, StreamResult(outputStream))
}
