package name.remal.gradle_plugins.plugins.assertj

import name.remal.gradle_plugins.dsl.*
import name.remal.gradle_plugins.dsl.extensions.all
import name.remal.gradle_plugins.dsl.extensions.applyLoggingTransitiveDependenciesExcludes
import name.remal.gradle_plugins.dsl.extensions.get
import name.remal.gradle_plugins.dsl.extensions.isCompilingSourceSet
import name.remal.gradle_plugins.plugins.common.CommonSettingsPlugin
import name.remal.gradle_plugins.plugins.java.JavaAnyPluginId
import name.remal.gradle_plugins.plugins.testing.TestSourceSetContainer
import name.remal.gradle_plugins.plugins.testing.TestSourceSetsPlugin
import name.remal.gradle_plugins.utils.getPredefinedDynamicVersionProperty
import org.gradle.api.Project
import org.gradle.api.artifacts.ConfigurationContainer
import org.gradle.api.artifacts.dsl.DependencyHandler
import org.gradle.api.tasks.TaskContainer
import org.gradle.api.tasks.compile.AbstractCompile

const val ASSERTJ_GENERATOR_CONFIGURATION_NAME = "assertjGenerator"
const val ASSERTJ_GENERATE_TASK_NAME = "assertjGenerate"

@Plugin(
    id = "name.remal.assertj-generator",
    description = "Plugin that configures AssertJ generator",
    tags = ["assertj"]
)
@ApplyPlugins(JavaAnyPluginId::class)
@ApplyPluginClasses(CommonSettingsPlugin::class, TestSourceSetsPlugin::class)
class AssertJGeneratorPlugin : BaseReflectiveProjectPlugin() {

    @CreateConfigurationsPluginAction
    fun ConfigurationContainer.`Create 'assertjGenerator' configuration`(project: Project, dependencyHandler: DependencyHandler, tasks: TaskContainer) {
        val assertjGeneratorConf = create(ASSERTJ_GENERATOR_CONFIGURATION_NAME) {
            it.description = "Classpath of AssertJ generator"
            it.applyLoggingTransitiveDependenciesExcludes()
            it.defaultDependencies {
                val version = project.getPredefinedDynamicVersionProperty("assertj-assertions-generator")
                it.add(dependencyHandler.create("org.assertj:assertj-assertions-generator:$version"))
            }
        }

        tasks.all(AssertJGenerate::class.java) {
            it.assertjGeneratorClasspath = assertjGeneratorConf
        }
    }

    @PluginAction(order = -100)
    fun TaskContainer.`Create 'assertjGenerate' task`() {
        create(ASSERTJ_GENERATE_TASK_NAME, AssertJGenerate::class.java)
    }

    @PluginAction
    fun TestSourceSetContainer.`Make all compile tasks of all testSourceSets depend on 'assertjGenerate' task`(tasks: TaskContainer) {
        val assertjGenerate = tasks[AssertJGenerate::class.java, ASSERTJ_GENERATE_TASK_NAME]
        all { testSourceSet ->
            testSourceSet.java.srcDir(assertjGenerate.outputDir)
            tasks.all(AbstractCompile::class.java) { compileTask ->
                if (compileTask.isCompilingSourceSet(testSourceSet)) {
                    compileTask.dependsOn(assertjGenerate)
                }
            }
        }
    }

}
