package name.remal.gradle_plugins.plugins.check_updates

import name.remal.debug
import name.remal.gradle_plugins.dsl.*
import name.remal.gradle_plugins.dsl.extensions.*
import name.remal.gradle_plugins.dsl.utils.DependencyNotation
import name.remal.gradle_plugins.dsl.utils.DependencyNotationMatcher
import name.remal.gradle_plugins.dsl.extensions.createWithNotation
import name.remal.gradle_plugins.plugins.dependencies.DependencyVersionsExtension
import name.remal.gradle_plugins.plugins.dependencies.DependencyVersionsPlugin
import org.gradle.api.DefaultTask
import org.gradle.api.Project
import org.gradle.api.artifacts.*
import org.gradle.api.plugins.HelpTasksPlugin.HELP_GROUP
import org.gradle.api.tasks.TaskAction
import org.gradle.api.tasks.TaskContainer
import kotlin.collections.set

const val CHECK_DEPENDENCY_UPDATES_TASK_NAME = "checkDependencyUpdates"
const val CHECK_DEPENDENCY_UPDATES_CONFIGURATION_NAME = "checkDependencyUpdates"

@Plugin(
    id = "name.remal.check-dependency-updates",
    description = "Plugin that provides task for discovering dependency updates",
    tags = ["versions", "dependency-updates"]
)
@ApplyPluginClasses(DependencyVersionsPlugin::class)
class CheckDependencyUpdatesPlugin : BaseReflectiveProjectPlugin() {

    @CreateConfigurationsPluginAction
    fun Project.`Create 'checkDependencyUpdates' configuration for project and all subprojects`() {
        allprojects { project ->
            project.configurations.getOrCreate(CHECK_DEPENDENCY_UPDATES_CONFIGURATION_NAME) {
                it.isCanBeConsumed = false
                it.description = "Configuration for check dependency updates"
                it.isTransitive = false
            }
        }
    }

    @PluginAction
    fun TaskContainer.`Create 'checkDependencyUpdates' task`() {
        create(CHECK_DEPENDENCY_UPDATES_TASK_NAME, CheckDependencyUpdates::class.java)
    }

}


val ConfigurationContainer.checkDependencyUpdates: Configuration get() = this[CHECK_DEPENDENCY_UPDATES_CONFIGURATION_NAME]


@BuildTask
class CheckDependencyUpdates : DefaultTask() {

    init {
        group = HELP_GROUP
        description = "Displays dependency updates for the project"

        onlyIf { !it.isParentProjectTaskWithSameNameInGraph }

        skipIfOffline()
    }

    var notCheckedDependencies: MutableSet<String> = mutableSetOf()

    var checkAllVersionsDependencies: MutableSet<String> = mutableSetOf()

    private val dependencyVersions = project[DependencyVersionsExtension::class.java]

    @TaskAction
    protected fun doCheckDependencyUpdates() {
        val skipMatchers = notCheckedDependencies.mapTo(mutableSetOf(), ::DependencyNotationMatcher)
        val allVersionsMatchers = checkAllVersionsDependencies.mapTo(mutableSetOf(), ::DependencyNotationMatcher)

        val resolvedNewVersionNotations = sortedMapOf<DependencyNotation, DependencyNotation>()

        project.allprojects.asSequence()
            .flatMap {
                sequenceOf(
                    it.buildscript.configurations,
                    it.configurations
                )
            }
            .plus(
                project.parents.asSequence().map {
                    it.buildscript.configurations
                }
            )
            .forEach forEachConfigurationContainer@{ configurations ->
                val processedNotations = hashSetOf<DependencyNotation>()
                val notations = configurations.asSequence()
                    .map {
                        it.copy().apply {
                            isCanBeResolved = true
                            dependencies.retainAll { it.doCheckForNewVersions }
                            dependencies.retainAll { dep -> skipMatchers.none { it.matches(dep) } }
                        }
                    }
                    .filter { it.dependencies.isNotEmpty() }
                    .onEach { logger.trace("Processing configuration: {}", it) }
                    .flatMap { it.resolvedConfiguration.lenientConfiguration.getFirstLevelModuleDependencies { it.doCheckForNewVersions }.asSequence() }
                    .map(ResolvedDependency::notation)
                    .filter(processedNotations::add)
                    .filter { notation -> skipMatchers.none { it.matches(notation) } }
                    .toSet()
                if (notations.isEmpty()) return@forEachConfigurationContainer

                notations.forEach forEachNotation@{ notation ->
                    sequenceOf(notation.withLatestVersion())
                        .flatMap {
                            sequenceOf(
                                it,
                                it.withoutClassifier(),
                                it.withoutExtension(),
                                it.withExtension("jar"),
                                it.withoutClassifier().withoutExtension(),
                                it.withoutClassifier().withExtension("jar"),
                                it.withoutClassifier().withExtension("pom")
                            )
                        }
                        .distinct()
                        .forEach forEachNotationVariant@{ notationToResolve ->
                            val dependency = project.dependencies.createWithNotation(notationToResolve) { it.isTransitive = false }
                            val resolveConf = configurations.detachedConfiguration(dependency).also {
                                it.isTransitive = false
                                it.resolutionStrategy {
                                    it.componentSelection {
                                        it.all { selection ->
                                            with(selection) {
                                                if (allVersionsMatchers.any { it.matches(candidate) }) {
                                                    return@all
                                                }

                                                dependencyVersions.getFirstInvalidToken(candidate.version)?.let { token ->
                                                    reject(token)
                                                    return@all
                                                }
                                            }
                                        }
                                    }
                                }
                            }

                            val lenientConfiguration = try {
                                resolveConf.resolvedConfiguration.lenientConfiguration
                            } catch (e: ResolveException) {
                                logger.debug(e)
                                return@forEachNotationVariant
                            }
                            lenientConfiguration.firstLevelModuleDependencies.forEach {
                                val resolvedNotation = it.notation
                                if (notation.version != resolvedNotation.version) {
                                    val keyNotation = notation.withoutClassifier().withoutExtension()
                                    val normalizedNotation = resolvedNotation.withoutClassifier().withoutExtension()
                                    val prevNotation = resolvedNewVersionNotations[keyNotation]
                                    if (prevNotation == null || prevNotation.compareVersions(resolvedNotation) < 0) {
                                        resolvedNewVersionNotations[keyNotation] = normalizedNotation
                                    }
                                }
                            }

                            return@forEachNotation
                        }
                }
            }

        resolvedNewVersionNotations.forEach { notation, resolvedNotation ->
            if (notation.group == resolvedNotation.group && notation.module == resolvedNotation.module) {
                logger.lifecycle("New dependency version: {}: {} -> {}", notation.withoutVersion(), notation.version, resolvedNotation.version)
            } else {
                logger.lifecycle("New dependency version: {} -> {}", notation, resolvedNotation)
            }
        }

        didWork = true
    }

    private val Dependency.doCheckForNewVersions: Boolean
        get() {
            if (this !is ExternalModuleDependency) {
                logDebug("Skipping as it's not instance of ExternalModuleDependency: {}", this)
                return false
            }
            if (isForce) {
                logDebug("Skipping as version of this dependency should be enforced: {}", this)
                return false
            }
            if (version == "+") {
                logDebug("Skipping as the latest version is used (+): {}", this)
                return false
            }
            return true
        }

}
