package name.remal.gradle_plugins.plugins.code_quality.findbugs.reporter

import name.remal.*
import name.remal.gradle_plugins.api.AutoService
import name.remal.gradle_plugins.dsl.extensions.calculateDestination
import name.remal.gradle_plugins.dsl.extensions.forClassLoader
import name.remal.gradle_plugins.dsl.extensions.setDefaultDestinationForTask
import name.remal.gradle_plugins.dsl.utils.getGradleLogger
import name.remal.gradle_plugins.plugins.code_quality.QualityTaskConsoleReporter
import name.remal.gradle_plugins.plugins.code_quality.QualityTaskHtmlReporter
import name.remal.gradle_plugins.plugins.code_quality.QualityTaskReporter
import name.remal.reflection.ExtendedURLClassLoader.LoadingOrder.THIS_FIRST
import org.gradle.api.plugins.quality.FindBugs
import org.gradle.api.plugins.quality.FindBugsReports
import org.gradle.api.reporting.ConfigurableReport
import org.gradle.api.reporting.Report.OutputType.DIRECTORY
import java.io.File
import javax.xml.transform.TransformerFactory

@AutoService(QualityTaskReporter::class)
class FindBugsReporter : QualityTaskConsoleReporter<FindBugs>, QualityTaskHtmlReporter<FindBugs> {

    companion object {
        private val logger = getGradleLogger(FindBugsReporter::class.java)
        private val findBugsReportsGetHtmlMethod = FindBugsReports::class.java.getMethod("getHtml").apply { isAccessible = true }
    }


    override fun configureTask(task: FindBugs) {
        task.reports.forEach { report ->
            report.isEnabled = report.name == "xml"
            if (report.isEnabled) {
                report.setDefaultDestinationForTask(task)
            }
        }
    }


    override fun printReportOf(task: FindBugs) {
        val xmlReportFile = task.reports.xml.destination.nullIf { !isFile } ?: return
        val messagesContainer = FindBugsXmlParser.parse(xmlReportFile).nullIf { messages.isEmpty() } ?: return
        val messages = messagesContainer.messages

        logger.error(
            "{} ({}) FindBugs violations were found in {} files",
            messages.size,
            messages.groupBy(FindBugsMessage::priority)
                .filterNotNullKeys()
                .mapValuesTo(mutableMapOf()) { it.value.size }
                .also { map -> arrayOf(1, 2, 3).forEach { map.putIfAbsent(it, 0) } }
                .toSortedMap()
                .values
                .joinToString(" / ")
            ,
            messagesContainer.filePaths.size
        )

        messages.forEach { message ->
            logger.error(
                "\n[priority {}] [{} | {}] {} ({}:{})\n    {}\n\n    {}",
                message.priority,
                message.categoryDescription ?: message.category,
                message.type,
                message.className,
                message.classFileName,
                message.line,
                message.description,
                message.detailText.replace("\n", "\n    ")
            )
        }
    }


    override fun writeHtmlReportFileOf(task: FindBugs) {
        // We have to use reflection to be compatible with Gradle 5
        val htmlReport: ConfigurableReport = findBugsReportsGetHtmlMethod.invoke(task.reports).uncheckedCast()
        val reportDestination = htmlReport.destination ?: htmlReport.calculateDestination(task)

        val xmlReportFile = task.reports.xml.destination ?: return
        if (!xmlReportFile.isFile) {
            reportDestination.forceDeleteRecursively()
            return
        }

        val htmlFile = if (DIRECTORY == htmlReport.outputType) File(reportDestination, "index.html") else reportDestination
        if (xmlReportFile.lastModified() <= htmlFile.lastModified()) return
        reportDestination.forceDeleteRecursively()

        htmlFile.createParentDirectories()
        task.findbugsClasspath.forClassLoader(THIS_FIRST) { classLoader ->
            val transformerUrl = arrayOf(
                "fancy.xsl",
                "plain.xsl",
                "default.xsl",
                "color.xsl"
            ).asSequence()
                .map(classLoader::getResource)
                .filterNotNull()
                .firstOrNull()
            if (transformerUrl != null) {
                TransformerFactory.newInstance().newTransformer(newTransformSource(transformerUrl))
                    .transform(xmlReportFile)
                    .into(newTransformResult(htmlFile))
            }
        }
    }

}

