package name.remal.gradle_plugins.plugins.code_quality.findbugs.reporter

import name.remal.*
import name.remal.gradle_plugins.dsl.utils.htmlToText
import org.jdom2.Element
import org.jdom2.input.SAXBuilder
import java.io.File
import java.util.*
import kotlin.collections.set

internal object FindBugsXmlParser {

    private class CacheItem(val lastModified: Long, val cachedValue: FindbugsXmlReport)

    private val cache = concurrentMapOf<File, CacheItem>()

    fun parse(file: File): FindbugsXmlReport {
        val lastModified = file.lastModified()
        cache[file]?.let { if (it.lastModified == lastModified) return it.cachedValue }
        synchronized(this) {
            cache[file]?.let { if (it.lastModified == lastModified) return it.cachedValue }
            val cachedValue = parseImpl(file)
            cache[file] = CacheItem(lastModified, cachedValue)
            return cachedValue
        }
    }

    private fun newSAXBuilder() = SAXBuilder().apply {
        setNoValidatingXMLReaderFactory()
        setNoOpEntityResolver()
    }

    private fun parseImpl(file: File): FindbugsXmlReport {
        val document = newSAXBuilder().build(file).clearNamespaces()

        val messages: List<FindBugsMessage>
        val bugInstances = document.getDescendants("BugInstance").toList()
        if (bugInstances.isNotEmpty()) {
            val categoryDescriptions = buildMap<String, String> {
                document.getDescendants("BugCategory").forEach { bugCategory ->
                    put(
                        bugCategory.getAttributeValue("category") ?: return@forEach,
                        bugCategory.getChild("Description")?.textTrim ?: return@forEach
                    )
                }
            }

            val detailHtmls = buildMap<String, String> {
                document.getDescendants("BugPattern").forEach { bugPattern ->
                    put(
                        bugPattern.getAttributeValue("type") ?: return@forEach,
                        bugPattern.getChild("Details")?.text?.replace("\r", "")?.trim('\n').default()
                    )
                }
            }
            val detailTexts = detailHtmls.mapValues { htmlToText(it.value) }

            messages = buildList<FindBugsMessage> {
                bugInstances.forEach { bugInstance ->
                    val category = bugInstance.getAttributeValue("category")
                    val type = bugInstance.getAttributeValue("type")
                    val sourceLine: Element? = run {
                        val sourceLines = listOf(
                            bugInstance.getChildren("SourceLine").reversed(),
                            bugInstance.getChildren("Method").reversed().map { it.getChildren("SourceLine").reversed() }.flatten(),
                            bugInstance.getChildren("Field").reversed().map { it.getChildren("SourceLine").reversed() }.flatten(),
                            bugInstance.getChildren("Class").reversed().map { it.getChildren("SourceLine").reversed() }.flatten()
                        ).flatten()
                        if (sourceLines.isEmpty()) return@run null
                        sourceLines.firstOrNull { !it.getAttributeValue("start").isNullOrEmpty() && it.getAttributeValue("primary") == "true" }?.let { return@run it }
                        sourceLines.firstOrNull { !it.getAttributeValue("start").isNullOrEmpty() }?.let { return@run it }
                        sourceLines.firstOrNull { it.getAttributeValue("primary") == "true" }?.let { return@run it }
                        return@run sourceLines.firstOrNull()
                    }
                    add(FindBugsMessage(
                        priority = bugInstance.getAttributeValue("priority")?.toIntOrNull(),
                        rank = bugInstance.getAttributeValue("rank")?.toIntOrNull(),
                        category = category,
                        categoryDescription = category?.let(categoryDescriptions::get),
                        type = type,
                        className = sourceLine?.getAttributeValue("classname"),
                        classFileName = sourceLine?.getAttributeValue("sourcefile"),
                        line = sourceLine?.getAttributeValue("start")?.toIntOrNull(),
                        description = bugInstance.getChild("LongMessage")?.textTrim.nullIfEmpty()
                            ?: bugInstance.getChild("ShortMessage")?.textTrim.nullIfEmpty()
                            ?: "",
                        detailText = type?.let(detailTexts::get).default(),
                        detailHtml = type?.let(detailHtmls::get).default()
                    ))
                }
            }.sorted()
        } else {
            messages = emptyList()
        }

        val filePaths: SortedSet<String> = document.getDescendants("FileStats")
            .map { it.getAttributeValue("path").default() }
            .filter { it.isNotEmpty() }
            .toSortedSet()

        return FindbugsXmlReport(
            messages = messages,
            filePaths = filePaths
        )
    }

}
