package name.remal.gradle_plugins.plugins.code_quality.sonar

import com.fasterxml.jackson.databind.ObjectMapper
import groovy.lang.Closure
import groovy.lang.Closure.DELEGATE_FIRST
import groovy.lang.DelegatesTo
import name.remal.*
import name.remal.gradle_plugins.dsl.BuildTask
import name.remal.gradle_plugins.dsl.extensions.*
import name.remal.gradle_plugins.dsl.utils.XML_PRETTY_OUTPUTTER
import name.remal.gradle_plugins.plugins.code_quality.setupQualityTaskReporters
import name.remal.gradle_plugins.plugins.code_quality.sonar.internal.SonarAnalysisResult
import name.remal.gradle_plugins.plugins.code_quality.sonar.internal.SonarLintInvoker
import name.remal.gradle_plugins.plugins.code_quality.sonar.internal.SonarSourceFile
import name.remal.gradle_plugins.plugins.code_quality.sonar.internal.impl.SonarLintInvokerImpl
import name.remal.json.data_format.DataFormatJSON.JSON_DATA_FORMAT
import org.gradle.api.Action
import org.gradle.api.GradleException
import org.gradle.api.file.FileCollection
import org.gradle.api.model.ObjectFactory
import org.gradle.api.reporting.Reporting
import org.gradle.api.tasks.*
import org.gradle.initialization.BuildCancellationToken
import org.jdom2.Document
import javax.inject.Inject
import name.remal.json.internal.Utils as JacksonUtils
import java.util.Optional as OptionalObj

@BuildTask
@CacheableTask
class SonarLint @Inject constructor(
    objectFactory: ObjectFactory,
    private val buildCancellationToken: BuildCancellationToken
) : SourceTask(), VerificationTask, Reporting<SonarLintReports>, SonarPropertiesMixin<SonarLint> {

    companion object {
        private val JSON_OBJECT_MAPPER: ObjectMapper by lazy { JSON_DATA_FORMAT.withIndention().objectMapper }
    }

    private var ignoreFailures: Boolean = false

    init {
        handleExtension { ignoreFailures = ignoreFailures || it.isIgnoreFailures }
    }

    override fun getIgnoreFailures() = ignoreFailures

    override fun setIgnoreFailures(ignoreFailures: Boolean) {
        this.ignoreFailures = ignoreFailures
    }


    private val reports: SonarLintReports = objectFactory.newInstance(SonarLintReports::class.java.taskReportContainerClass, this).apply {
        xml.isEnabled = true
    }

    @Nested
    override fun getReports() = reports

    override fun reports(@DelegatesTo(SonarLintReports::class, strategy = DELEGATE_FIRST) closure: Closure<Any>) = reports.configureWith(closure)
    override fun reports(configureAction: Action<in SonarLintReports>) = reports.configureWith(configureAction)


    @get:Input
    @get:Optional
    var excludedMessages: MutableSet<String> = sortedSetOf()
        set(value) {
            field = value.toSortedSet()
        }

    fun excludeMessage(value: String) = excludeMessages(value)

    fun excludeMessages(vararg values: String) = apply {
        excludedMessages.addAll(values)
    }

    fun excludeMessages(values: Iterable<String>) = apply {
        excludedMessages.addAll(values)
    }

    init {
        handleExtension {
            it.excludes.also {
                exclude(it.sources)
                excludeMessages(it.messages)
            }
        }
    }


    @get:Input
    var isTestSources: Boolean = false


    @get:Input
    @get:Optional
    override var sonarProperties: MutableMap<String, String?> = sortedMapOf()
        set(value) {
            field = value.toSortedMap()
        }

    init {
        handleExtension { sonarProperties.putAll(it.sonarProperties) }
    }


    @get:Classpath
    var sonarCoreClasspath: FileCollection = project.emptyFileCollection()

    @get:Classpath
    var sonarPluginsClasspath: FileCollection = project.emptyFileCollection()


    @TaskAction
    fun doAnalyze() {
        reports.forEach { it.setDefaultDestinationForTask(this) }

        val inputFiles = mutableListOf<SonarSourceFile>()
        getSource().visitFiles { fileDetails ->
            inputFiles.add(SonarSourceFile(
                fileDetails.file,
                fileDetails.path,
                isTestSources
            ))
        }

        lateinit var analysisResult: SonarAnalysisResult
        sonarCoreClasspath.forClassLoader { classLoader ->
            classLoader.forInstantiatedWithPropagatedPackage(SonarLintInvoker::class.java, SonarLintInvokerImpl::class.java) { invoker ->
                analysisResult = invoker.invoke(inputFiles, excludedMessages, sonarProperties.filterNotNullValues(), project.projectDir, sonarPluginsClasspath, logger, buildCancellationToken)
            }
        }

        reports.json.nullIf { !isEnabled }?.also { report ->
            JSON_OBJECT_MAPPER.writeValue(report.destination.createParentDirectories(), analysisResult)
        }
        reports.xml.nullIf { !isEnabled }?.also { report ->
            val xmlElement = analysisResult.toXmlElement()
            val xmlDocument = Document(xmlElement)
            report.destination.createParentDirectories().outputStream().use { outputStream ->
                XML_PRETTY_OUTPUTTER.output(xmlDocument, outputStream)
            }
        }

        didWork = true

        if (!getIgnoreFailures() && analysisResult.issues.isNotEmpty()) {
            val errorMessage = "%d SonarLint violations were found in %d files".format(analysisResult.issues.size, analysisResult.scannedFileCount)
            throw GradleException(errorMessage)
        }
    }


    init {
        setupQualityTaskReporters(this)
    }


    private fun handleExtension(handler: (extension: SonarLintExtension) -> Unit) {
        onlyIf {
            it.project.extensions.findByType(SonarLintExtension::class.java)?.let(handler)
            return@onlyIf true
        }
    }

}
