package name.remal.gradle_plugins.plugins.code_quality.sonar

import name.remal.gradle_plugins.api.BuildTimeConstants.getStringProperties
import name.remal.gradle_plugins.api.BuildTimeConstants.getStringProperty
import name.remal.gradle_plugins.dsl.*
import name.remal.gradle_plugins.dsl.extensions.*
import name.remal.gradle_plugins.plugins.common.FilteringSettingsPlugin
import name.remal.gradle_plugins.plugins.java.JavaAnyPluginId
import name.remal.gradle_plugins.plugins.java.JavaSettingsPlugin
import name.remal.gradle_plugins.plugins.kotlin.KotlinAnyPluginId
import name.remal.gradle_plugins.plugins.kotlin.KotlinJsSettingsPlugin
import name.remal.gradle_plugins.plugins.kotlin.KotlinJvmSettingsPlugin
import name.remal.gradle_plugins.plugins.testing.TestSourceSetContainer
import name.remal.nullIfEmpty
import org.gradle.api.Project
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ConfigurationContainer
import org.gradle.api.artifacts.ModuleDependency
import org.gradle.api.artifacts.dsl.DependencyHandler
import org.gradle.api.artifacts.dsl.RepositoryHandler
import org.gradle.api.artifacts.repositories.MavenArtifactRepository
import org.gradle.api.plugins.ExtensionContainer
import org.gradle.api.plugins.JavaBasePlugin.CHECK_TASK_NAME
import org.gradle.api.tasks.SourceSet
import org.gradle.api.tasks.SourceSet.MAIN_SOURCE_SET_NAME
import org.gradle.api.tasks.SourceSet.TEST_SOURCE_SET_NAME
import org.gradle.api.tasks.SourceSetContainer
import org.gradle.api.tasks.TaskContainer
import java.io.File

@Plugin(
    id = "name.remal.sonarlint",
    description = "Plugin that executes SonarLint checks without SonarQube server.",
    tags = ["sonar", "sonarlint"]
)
@ApplyPluginClasses(
    FilteringSettingsPlugin::class,
    JavaSettingsPlugin::class,
    KotlinJvmSettingsPlugin::class,
    KotlinJsSettingsPlugin::class
)
@SimpleTestAdditionalGradleScript("""
    sonarlint {
        ignoreFailures = false
        excludes {
            message 'squid:S2094'
        }
    }
""")
class SonarLintPlugin : BaseReflectiveProjectPlugin() {

    companion object {
        private val SONARSOURCE_REPOSITORY = getStringProperty("sonarsource.repository")

        private val SONARLINT_DEFAULT_VERSION = getStringProperty("sonarlint-core.latest-version")
        private val SONARLINT_VERSION_PROPERTY = "sonarlint-core.version"

        private val SONAR_PLUGIN_PROPERTIES = getStringProperties("sonar-*-plugin.*")
    }

    @CreateExtensionsPluginAction
    fun ExtensionContainer.`Create 'sonarlint' extension`(project: Project) {
        create("sonarlint", SonarLintExtension::class.java).also { extension ->
            extension.toolVersion = project.findProperty(SONARLINT_VERSION_PROPERTY).unwrapProviders()?.toString() ?: SONARLINT_DEFAULT_VERSION
            extension.reportsDir = project.reportsDir
            extension.sourceSets = mutableListOf()
            project.withPlugin(JavaAnyPluginId) { _ ->
                extension.sourceSets = project.java.sourceSets
            }
        }
    }

    @PluginAction(order = -110)
    fun RepositoryHandler.`Add SonarSource repository`() {
        if (none { it is MavenArtifactRepository && it.url.toString() == SONARSOURCE_REPOSITORY }) {
            maven {
                it.name = "SonarSource"
                it.setUrl(SONARSOURCE_REPOSITORY)
            }
        }
    }

    @PluginAction(order = -100)
    fun ConfigurationContainer.`Setup 'sonarlintCore' configuration default dependencies`(extensions: ExtensionContainer, dependencyHandler: DependencyHandler) {
        sonarlintCore.defaultDependencies { dependencies ->
            val toolVersion = extensions[SonarLintExtension::class.java].toolVersion
            dependencies.add(dependencyHandler.create("org.sonarsource.sonarlint.core:sonarlint-core:$toolVersion"))
            dependencies.add(dependencyHandler.create("commons-lang:commons-lang:2.6"))
            dependencies.add(dependencyHandler.create("com.google.code.findbugs:jsr305:3.0.2"))
        }
    }

    @PluginAction(order = -90)
    @WithPlugins(JavaAnyPluginId::class)
    fun ConfigurationContainer.`Add java Sonar plugin dependency`(project: Project, dependencyHandler: DependencyHandler) {
        addSonarPluginDependency(project, dependencyHandler, "java")
    }

    @PluginAction(order = -90)
    @WithPlugins(KotlinAnyPluginId::class)
    fun ConfigurationContainer.`Add kotlin Sonar plugin dependency`(project: Project, dependencyHandler: DependencyHandler) {
        addSonarPluginDependency(project, dependencyHandler, "kotlin")
    }

    @PluginAction(order = -90)
    fun ConfigurationContainer.`Add html, javascript, xml Sonar plugin dependencies`(project: Project, dependencyHandler: DependencyHandler) {
        addSonarPluginDependency(project, dependencyHandler, "html", "javascript", "xml")
    }

    @PluginAction
    fun TaskContainer.`Configure SonarLint tasks' Sonar classpath`(configurations: ConfigurationContainer) {
        all(SonarLint::class.java) { task ->
            task.sonarCoreClasspath = configurations.sonarlintCore
            task.sonarPluginsClasspath = configurations.sonarlintPlugins
        }
    }


    @PluginAction
    @WithPlugins(JavaAnyPluginId::class)
    fun `Lazily create analyze task for each source-set`(sourceSets: SourceSetContainer, tasks: TaskContainer, extensions: ExtensionContainer, project: Project) {
        fun SourceSet.isTest(): Boolean {
            if (name == MAIN_SOURCE_SET_NAME) {
                return false
            }
            if (name == TEST_SOURCE_SET_NAME) {
                return true
            }
            extensions.findByType(TestSourceSetContainer::class.java)?.let {
                if (this in it) {
                    return true
                }
            }
            return false
        }

        fun SourceSet.isNotTest() = !isTest()

        sourceSets.all { sourceSet ->
            tasks.registerCompatible(sourceSet.sonarlintTaskName, SonarLint::class.java) { task ->
                task.dependsOn(sourceSet.classesTaskName)
                task.source(sourceSet.allSource)

                if (sourceSet.isTest()) {
                    task.isTestSources = true
                }

                task.onlyIf {
                    sourceSet.output.files.filter(File::exists).map(File::getAbsoluteFile).joinToString(",").nullIfEmpty()?.also { task.sonarProperty("sonar.java.binaries", it) }
                    sourceSet.compileClasspath.files.filter(File::exists).map(File::getAbsoluteFile).joinToString(",").nullIfEmpty()?.also { task.sonarProperty("sonar.java.libraries", it) }

                    if (sourceSet.isTest()) {
                        task.isTestSources = true
                        task.sonarProperty("sonar.java.test.binaries", task.sonarProperties["sonar.java.binaries"])
                        task.sonarProperty("sonar.java.test.libraries", task.sonarProperties["sonar.java.libraries"])

                    } else {
                        val allBinaries = mutableSetOf<File>()
                        val allLibraries = mutableSetOf<File>()
                        sourceSets.filter { it.isTest() }.forEach {
                            it.output.files.filter(File::exists).mapTo(allBinaries, File::getAbsoluteFile)
                            it.compileClasspath.files.filter(File::exists).mapTo(allLibraries, File::getAbsoluteFile)
                        }
                        allBinaries.joinToString(",").nullIfEmpty()?.also { task.sonarProperty("sonar.java.test.binaries", it) }
                        allLibraries.joinToString(",").nullIfEmpty()?.also { task.sonarProperty("sonar.java.test.libraries", it) }
                    }

                    return@onlyIf true
                }

                task.dependsOnIf(sourceSet::isNotTest) {
                    sourceSets.asSequence()
                        .filter { it.isTest() }
                        .mapNotNull(SourceSet::getClassesTaskName)
                        .mapNotNull(tasks::findByName)
                        .toList()
                }
            }
        }

        tasks.all(CHECK_TASK_NAME) { checkTask ->
            checkTask.dependsOn {
                sourceSets.toMutableList()
                    .apply { retainAll(extensions[SonarLintExtension::class.java].sourceSets) }
                    .mapNotNull(SourceSet::sonarlintTaskName)
                    .mapNotNull(tasks::findByName)
            }
        }
    }


    private val ConfigurationContainer.sonarlintCore: Configuration
        get() = this.getOrCreate("sonarlintCore") {
            it.description = "SonarLint core"
        }

    private val ConfigurationContainer.sonarlintPlugins: Configuration
        get() = this.getOrCreate("sonarlintPlugins") {
            it.description = "SonarLint plugins"
            it.dependencies.all { if (it is ModuleDependency) it.isTransitive = false }
        }


    private fun ConfigurationContainer.addSonarPluginDependency(project: Project, dependencyHandler: DependencyHandler, vararg sonarPlugins: String) {
        sonarPlugins.forEach { sonarPlugin ->
            val latestVersion = SONAR_PLUGIN_PROPERTIES["sonar-$sonarPlugin-plugin.latest-version"] ?: throw IllegalStateException("Last version isn't defined for '$sonarPlugin' Sonar plugin")
            val version = project.findProperty("sonar-$sonarPlugin-plugin.version").unwrapProviders()?.toString() ?: latestVersion
            val notation = (SONAR_PLUGIN_PROPERTIES["sonar-$sonarPlugin-plugin.dependency-notation"] ?: throw IllegalStateException("Notation isn't defined for '$sonarPlugin' Sonar plugin"))
                .let {
                    val tokens = it.split(':').toMutableList()
                    if (tokens.size <= 2) {
                        tokens.add(version)
                    } else {
                        tokens[2] = version
                    }
                    return@let tokens.joinToString(":")
                }
            val dependency = dependencyHandler.create(notation)
            if (dependency is ModuleDependency) dependency.isTransitive = false
            sonarlintPlugins.dependencies.add(dependency)
        }
    }

}


val SourceSet.sonarlintTaskName: String get() = getTaskName("sonarlint", null)
