package name.remal.gradle_plugins.plugins.code_quality.sonar.internal.impl

import name.remal.*
import name.remal.gradle_plugins.plugins.code_quality.sonar.internal.*
import org.gradle.api.logging.Logger
import org.gradle.initialization.BuildCancellationToken
import org.sonarsource.sonarlint.core.StandaloneSonarLintEngineImpl
import org.sonarsource.sonarlint.core.client.api.common.RuleKey
import org.sonarsource.sonarlint.core.client.api.common.analysis.ClientInputFile
import org.sonarsource.sonarlint.core.client.api.common.analysis.IssueListener
import org.sonarsource.sonarlint.core.client.api.common.analysis.IssueLocation
import org.sonarsource.sonarlint.core.client.api.standalone.StandaloneAnalysisConfiguration
import org.sonarsource.sonarlint.core.client.api.standalone.StandaloneGlobalConfiguration
import java.io.File

internal class SonarLintInvokerImpl : SonarLintInvoker {

    override fun invoke(sourceFiles: Iterable<SonarSourceFile>, excludedMessages: Iterable<String>, sonarProperties: Map<String, String>, projectDir: File, pluginFiles: Iterable<File>, logger: Logger, buildCancellationToken: BuildCancellationToken): SonarAnalysisResult {
        return SONARLINT_TEMP_DIRS_WITH_CLEANUP.withTempDir { tempDir ->

            val logOutput = GradleLogOutput(logger)
            val progressMonitor = GradleProgressMonitor(logger, buildCancellationToken)
            val homeDir = tempDir.resolve("home").createDirectories()
            val workDir = tempDir.resolve("work").createDirectories()
            val engineConfig = StandaloneGlobalConfiguration.builder()
                .apply {
                    pluginFiles.forEach { file ->
                        val pluginUrl = file.toURI().toURL()
                        logger.debug("Use Sonar plugin: {}", pluginUrl)
                        addPlugin(pluginUrl)
                    }
                }
                .setSonarLintUserHome(homeDir.toPath())
                .setWorkDir(workDir.toPath())
                .setLogOutput(logOutput)
                .build()

            val engine = StandaloneSonarLintEngineImpl(engineConfig)
            try {

                val analysisConfig = StandaloneAnalysisConfiguration(projectDir.toPath(), workDir.toPath(), sourceFiles.mapToClientInputFiles(), sonarProperties, excludedMessages.map(RuleKey::parse), emptyList())
                logger.debug("Starting Sonar analysis with configuration:\n{}", analysisConfig)

                val issues = mutableListOf<SonarIssue>().asSynchronized()
                val issueListener = IssueListener { issue ->
                    val details = engine.getRuleDetails(issue.ruleKey).orNull
                    issues.add(SonarIssue(
                        location = issue.toSourceFileLocation(),
                        ruleKey = issue.ruleKey,
                        severity = issue.severity,
                        type = issue.type,
                        description = issue.ruleName,
                        detailHtml = details?.htmlDescription,
                        flows = issue.flows().default().map {
                            SonarFlow(
                                locations = it.locations().map(IssueLocation::toSourceFileLocation)
                            )
                        }
                    ))
                }

                val result = engine.analyze(analysisConfig, issueListener, logOutput, progressMonitor)

                return@withTempDir SonarAnalysisResult(
                    scannedFileCount = result.indexedFileCount(),
                    failedAnalysisFiles = result.failedAnalysisFiles().map(ClientInputFile::getClientObject),
                    definedSourceFileLanguages = result.languagePerFile()
                        .mapKeys { it.key.getClientObject<SonarSourceFile>() }
                        .mapValues { it.value.nullIfEmpty() }
                        .filterNotNullValues()
                        .map { SonarSourceFileLanguage(it.key.absolutePath, it.key.relativePath, it.value) }
                    ,
                    issues = issues.sorted()
                )

            } finally {
                engine.stop()
            }

        }
    }

}
