package name.remal.gradle_plugins.plugins.code_quality.sonar.reporter

import com.fasterxml.jackson.databind.ObjectMapper
import name.remal.gradle_plugins.api.AutoService
import name.remal.gradle_plugins.dsl.extensions.setDefaultDestinationForTask
import name.remal.gradle_plugins.dsl.utils.getGradleLogger
import name.remal.gradle_plugins.plugins.code_quality.QualityTaskConsoleReporter
import name.remal.gradle_plugins.plugins.code_quality.QualityTaskReporter
import name.remal.gradle_plugins.plugins.code_quality.findbugs.reporter.FindBugsReporter
import name.remal.gradle_plugins.plugins.code_quality.sonar.SonarLint
import name.remal.gradle_plugins.plugins.code_quality.sonar.internal.SonarAnalysisResult
import name.remal.json.data_format.DataFormatJSON.JSON_DATA_FORMAT
import name.remal.nullIf
import name.remal.nullIfEmpty

@AutoService(QualityTaskReporter::class)
class SonarLintReporter : QualityTaskConsoleReporter<SonarLint> {

    companion object {
        private val logger = getGradleLogger(FindBugsReporter::class.java)
        private val JSON_OBJECT_MAPPER: ObjectMapper by lazy { JSON_DATA_FORMAT.objectMapper }
    }


    override fun configureTask(task: SonarLint) {
        task.reports.json.also { report ->
            report.isEnabled = true
            report.setDefaultDestinationForTask(task)
        }
    }


    override fun printReportOf(task: SonarLint) {
        val jsonReportFile = task.reports.json.destination.nullIf { !isFile } ?: return
        val analysisResult = JSON_OBJECT_MAPPER.readValue(jsonReportFile, SonarAnalysisResult::class.java)
        if (analysisResult.issues.isEmpty()) return

        logger.error("{} SonarLint violations were found in {} files", analysisResult.issues.size, analysisResult.scannedFileCount)

        analysisResult.issues.forEach { issue ->
            logger.error(buildString {
                append('\n')
                append('[').append(issue.severity).append(']')
                append(' ')
                issue.type.nullIfEmpty()?.also {
                    append('[').append(it).append(']')
                    append(' ')
                }
                issue.location.absolutePath.nullIfEmpty()?.also {
                    append(it)
                    append(": ")
                    append('(').append(issue.location.startLine ?: 1).append(", ").append(issue.location.startLineOffset ?: 1).append(')')
                    append(": ")
                }
                append(issue.ruleKey)
                append(": ")
                append(issue.description)
                issue.detailText?.also {
                    append("\n    ")
                    append(it.replace("\n", "\n    "))
                }
            })
        }
    }

}
