package name.remal.gradle_plugins.plugins.code_quality.spotbugs.reporter

import com.github.spotbugs.SpotBugsTask
import name.remal.*
import name.remal.gradle_plugins.api.AutoService
import name.remal.gradle_plugins.dsl.extensions.calculateDestination
import name.remal.gradle_plugins.dsl.extensions.forClassLoader
import name.remal.gradle_plugins.dsl.extensions.setDefaultDestinationForTask
import name.remal.gradle_plugins.dsl.utils.getGradleLogger
import name.remal.gradle_plugins.plugins.code_quality.QualityTaskConsoleReporter
import name.remal.gradle_plugins.plugins.code_quality.QualityTaskHtmlReporter
import name.remal.gradle_plugins.plugins.code_quality.QualityTaskReporter
import name.remal.gradle_plugins.plugins.code_quality.findbugs.reporter.FindBugsMessage
import name.remal.gradle_plugins.plugins.code_quality.findbugs.reporter.FindBugsXmlParser
import name.remal.reflection.ExtendedURLClassLoader.LoadingOrder.THIS_FIRST
import org.gradle.api.reporting.ConfigurableReport
import org.gradle.api.reporting.Report.OutputType.DIRECTORY
import java.io.File
import javax.xml.transform.TransformerFactory

@AutoService(QualityTaskReporter::class)
class SpotBugsReporter : QualityTaskConsoleReporter<SpotBugsTask>, QualityTaskHtmlReporter<SpotBugsTask> {

    companion object {
        private val logger = getGradleLogger(SpotBugsReporter::class.java)
    }


    override fun configureTask(task: SpotBugsTask) {
        task.reports.forEach { report ->
            report.isEnabled = report.name == "xml"
            if (report.isEnabled) {
                report.setDefaultDestinationForTask(task)
            }
        }
    }


    override fun printReportOf(task: SpotBugsTask) {
        val xmlReportFile = task.reports.xml.destination.nullIf { !isFile } ?: return
        val messagesContainer = FindBugsXmlParser.parse(xmlReportFile).nullIf { messages.isEmpty() } ?: return
        val messages = messagesContainer.messages

        logger.error(
            "{} ({}) SpotBugs violations were found in {} files",
            messages.size,
            messages.groupBy(FindBugsMessage::priority)
                .filterNotNullKeys()
                .mapValuesTo(mutableMapOf()) { it.value.size }
                .also { map -> arrayOf(1, 2, 3).forEach { map.putIfAbsent(it, 0) } }
                .toSortedMap()
                .values
                .joinToString(" / ")
            ,
            messagesContainer.filePaths.size
        )

        messages.forEach { message ->
            logger.error(
                "\n[priority {}] [{} | {}] {} ({}:{})\n    {}\n\n    {}",
                message.priority,
                message.categoryDescription ?: message.category,
                message.type,
                message.className,
                message.classFileName,
                message.line,
                message.description,
                message.detailText.replace("\n", "\n    ")
            )
        }
    }


    override fun writeHtmlReportFileOf(task: SpotBugsTask) {
        val htmlReport: ConfigurableReport = task.reports.html
        val reportDestination = htmlReport.destination ?: htmlReport.calculateDestination(task)

        val xmlReportFile = task.reports.xml.destination ?: return
        if (!xmlReportFile.isFile) {
            reportDestination.forceDeleteRecursively()
            return
        }

        val htmlFile = if (DIRECTORY == htmlReport.outputType) File(reportDestination, "index.html") else reportDestination
        if (xmlReportFile.lastModified() <= htmlFile.lastModified()) return
        reportDestination.forceDeleteRecursively()

        htmlFile.createParentDirectories()
        task.spotbugsClasspath.forClassLoader(THIS_FIRST) { classLoader ->
            arrayOf(
                "fancy.xsl",
                "plain.xsl",
                "default.xsl",
                "color.xsl"
            ).asSequence()
                .map(classLoader::getResource)
                .filterNotNull()
                .firstOrNull()
                ?.let { transformerUrl ->
                    TransformerFactory.newInstance().newTransformer(newTransformSource(transformerUrl))
                        .transform(xmlReportFile)
                        .into(newTransformResult(htmlFile))
                }
        }
    }

}

