package name.remal.gradle_plugins.plugins.common

import name.remal.createParentDirectories
import name.remal.forceDelete
import name.remal.gradle_plugins.dsl.*
import name.remal.gradle_plugins.dsl.extensions.*
import name.remal.gradle_plugins.plugins.vcs.*
import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.plugins.BasePlugin.ASSEMBLE_TASK_NAME
import org.gradle.api.plugins.ExtensionContainer
import org.gradle.api.plugins.JavaBasePlugin.BUILD_TASK_NAME
import org.gradle.api.plugins.JavaBasePlugin.CHECK_TASK_NAME
import org.gradle.api.plugins.JavaPlugin.TEST_TASK_NAME
import org.gradle.api.publish.maven.tasks.AbstractPublishToMaven
import org.gradle.api.tasks.TaskContainer
import java.io.File
import java.lang.System.getenv

@Plugin(
    id = "name.remal.rebuild-if-dependencies-are-changed",
    description = "Creates 'createDependenciesHashTag' and handles 'PUBLISH_ONLY_IF_DEPENDENCIES_ARE_CHANGED' environment variable.",
    tags = ["dependencies"],
    isHidden = true
)
@ApplyPluginClasses(AutoVcsVersionPlugin::class)
@EnvironmentVariable("PUBLISH_ONLY_IF_DEPENDENCIES_ARE_CHANGED", description = "Publish only if dependencies are changed")
class RebuildIfDependenciesAreChangedPlugin : BaseReflectiveProjectPlugin() {

    private val PUBLISH_ONLY_IF_DEPENDENCIES_ARE_CHANGED: Boolean by lazy {
        getenv("PUBLISH_ONLY_IF_DEPENDENCIES_ARE_CHANGED")?.toBoolean() == true
            || System.getProperty("PUBLISH_ONLY_IF_DEPENDENCIES_ARE_CHANGED")?.toBoolean() == true
    }

    @PluginAction
    fun TaskContainer.`Create 'createDependenciesHashTag' task`() {
        create("createDependenciesHashTag", CreateDependenciesHashTag::class.java) { task ->
            task.failIfTagExists = false
            task.failIfTagExistsOnCurrentCommit = false

            task.project.rootProject.allprojects { project ->
                project.tasks.all(CreateProjectVersionTag::class.java) { dependentTask ->
                    dependentTask.dependsOn(task)
                }
            }

            task.mustRunAfter {
                task.project.rootProject.allprojects.asSequence()
                    .map(Project::getTasks)
                    .flatMap {
                        sequenceOf(ASSEMBLE_TASK_NAME, TEST_TASK_NAME, CHECK_TASK_NAME, BUILD_TASK_NAME)
                            .mapNotNull(it::findByName)
                    }
                    .toList()
            }


            fun forEachDependentTask(action: (dependentTask: Task) -> Unit) {
                arrayOf<Class<out Task>>(
                    CreateProjectVersionTag::class.java,
                    BaseCreateTagTask::class.java,
                    AbstractPublishToMaven::class.java
                ).forEach { dependentTaskClass ->
                    val processedTasks = hashSetOf<Task>()
                    task.project.rootProject.allprojects { project ->
                        project.tasks.all(dependentTaskClass) { dependentTask ->
                            if (dependentTask !is CreateDependenciesHashTag) {
                                if (processedTasks.add(dependentTask)) {
                                    action(dependentTask)
                                }
                            }
                        }
                    }
                }
            }

            forEachDependentTask { dependentTask ->
                dependentTask.mustRunAfter(task)
            }

            task.doFirst {
                task.project.doneFile.also { doneFile ->
                    task.logDebug("Removing doneFile: {}", doneFile)
                    doneFile.forceDelete()
                }
            }
            task.onTagCreated {
                task.project.doneFile.also { doneFile ->
                    task.logDebug("Creating doneFile: {}", doneFile)
                    doneFile.createParentDirectories().writeText("true")
                }
            }
            if (PUBLISH_ONLY_IF_DEPENDENCIES_ARE_CHANGED) {
                task.onTagNotCreated {
                    forEachDependentTask { dependentTask ->
                        task.logWarn("Disabling {}", dependentTask)
                        dependentTask.enabled = false
                    }
                }
            }
        }
    }

    @PluginActionsGroup(order = Int.MAX_VALUE)
    inner class `If PUBLISH_ONLY_IF_DEPENDENCIES_ARE_CHANGED environment variable equals to true` {

        @PluginCondition(isHidden = true)
        private fun condition(): Boolean {
            return PUBLISH_ONLY_IF_DEPENDENCIES_ARE_CHANGED
        }

        @PluginAction
        fun Project.`Disable publish tasks if project version tag hasn't been created`(extensions: ExtensionContainer) {
            val autoVcsVersion = extensions[AutoVcsVersionExtension::class.java]
            val vcsOperations = extensions[VcsOperationsExtension::class.java]
            rootProject.allprojects { project ->
                project.tasks.all(AbstractPublishToMaven::class.java) { publishTask ->
                    publishTask.onlyIf {
                        val prefixes: List<String> = autoVcsVersion.versionTagPrefixes.let {
                            if (it.isEmpty()) {
                                listOf("")
                            } else {
                                it.sortedDescending()
                            }
                        }

                        val tagNames = vcsOperations.getAllTagNames()

                        return@onlyIf prefixes.any { prefix ->
                            "$prefix${publishTask.project.version}" in tagNames
                        }
                    }
                }
            }
        }

        @PluginAction(order = Int.MAX_VALUE)
        fun ExtensionContainer.`Increment project version by 1`(project: Project) {
            project.doneFile.let { doneFile ->
                if (!doneFile.isFile) {
                    logger.debug("doneFile doesn't exist: {}", doneFile)
                    logger.lifecycle("Incrementing VCS version...")
                    this[AutoVcsVersionExtension::class.java].incrementVersionBy++

                } else {
                    logger.debug("doneFile exists: {}", doneFile)
                }
                Unit
            }
        }

    }


    private val Project.doneFile: File get() = buildDir.resolve(CreateDependenciesHashTag::class.java.simpleName + ".done")

}
