package name.remal.gradle_plugins.plugins.dependencies

import name.remal.asSynchronized
import name.remal.concurrentMapOf
import name.remal.escapeRegex
import name.remal.findAll
import name.remal.gradle_plugins.api.BuildTimeConstants.getClassSimpleName
import name.remal.gradle_plugins.dsl.Extension
import name.remal.gradle_plugins.dsl.extensions.createWithNotation
import name.remal.gradle_plugins.dsl.utils.DependencyNotation
import name.remal.gradle_plugins.dsl.utils.getGradleLogger
import name.remal.gradle_plugins.dsl.utils.parseDependencyNotation
import name.remal.version.Version
import org.gradle.api.artifacts.ConfigurationContainer
import org.gradle.api.artifacts.dsl.DependencyHandler
import org.gradle.api.artifacts.dsl.RepositoryHandler
import org.gradle.api.artifacts.repositories.MavenArtifactRepository
import org.gradle.internal.resolve.ModuleVersionNotFoundException
import java.net.URI
import java.util.Comparator.reverseOrder
import java.util.concurrent.ConcurrentMap
import kotlin.text.RegexOption.IGNORE_CASE

@Extension
class DependencyVersionsExtension(
    private val dependencies: DependencyHandler,
    private val configurations: ConfigurationContainer,
    private val repositories: RepositoryHandler
) {

    companion object {
        private val logger = getGradleLogger(DependencyVersionsExtension::class.java)
        private val allCaches: ConcurrentMap<AllCachesKey, Caches> = concurrentMapOf()
    }


    var invalidVersionTokens: MutableSet<String> = mutableSetOf("alpha", "beta", "rc", "cr", "m", "redhat", "b", "pr", "dev", "snapshot")
        set(value) {
            field = value.toSortedSet(reverseOrder())
        }

    fun getFirstInvalidToken(version: String): String? {
        return invalidVersionTokens.firstOrNull { Regex(".*[._-]${escapeRegex(it)}[._-]?\\d*(\\.\\d*)*([._-].*)?", IGNORE_CASE).matches(version) }
    }

    fun resolveLatestVersion(notation: String): String {
        return caches.resolveLatestVersionCache.computeIfAbsent(parseDependencyNotation(notation).withLatestVersion()) { dependencyNotation ->
            val dependency = dependencies.createWithNotation(dependencyNotation)
            dependency.isTransitive = false
            configurations.detachedConfiguration().let { conf ->
                conf.isTransitive = false
                conf.resolutionStrategy {
                    it.componentSelection {
                        it.all { selection ->
                            getFirstInvalidToken(selection.candidate.version)?.let { token ->
                                selection.reject(token)
                            }
                        }
                    }
                }

                conf.dependencies.add(dependency)

                return@computeIfAbsent conf.resolvedConfiguration.firstLevelModuleDependencies.first().moduleVersion.also {
                    logger.info("Dependency {} resolved with {} version", dependencyNotation, it)
                }
            }
        }
    }


    fun resolveAllVersions(notation: String): List<String> {
        return caches.allVersionsCache.computeIfAbsent(parseDependencyNotation(notation).withLatestVersion()) { dependencyNotation ->
            val dependency = dependencies.createWithNotation(dependencyNotation)
            dependency.isTransitive = false
            val result = mutableSetOf<String>().asSynchronized()
            configurations.detachedConfiguration().also { conf ->
                conf.isTransitive = false
                conf.dependencies.add(dependency)
                conf.resolutionStrategy {
                    it.componentSelection {
                        it.all { selection ->
                            with(selection) {
                                val version = candidate.version

                                getFirstInvalidToken(version)?.let { token ->
                                    reject(token)
                                    return@with
                                }

                                result.add(version)

                                reject(DependencyVersionsExtension::resolveAllVersions.name)
                            }
                        }
                    }
                }
                try {
                    conf.resolve()
                } catch (e: Exception) {
                    if (e.isFullyIgnorable) {
                        // do nothing
                    } else if (e.isIgnorable) {
                        logger.warn(e.toString())
                        // do nothing
                    } else {
                        throw e
                    }
                }
            }

            return@computeIfAbsent result.toSortedVersions()
        }
    }


    private val caches: Caches
        get() {
            val uris = hashSetOf<URI>()
            repositories.forEach {
                if (it !is MavenArtifactRepository) {
                    return Caches()
                }
                uris.add(it.url)
            }

            return allCaches.computeIfAbsent(AllCachesKey(invalidVersionTokens.toSet(), uris), { Caches() })
        }

    private data class AllCachesKey(
        val invalidVersionTokens: Set<String>,
        val repositoriesUris: Set<URI>
    )


    private class Caches {
        val resolveLatestVersionCache: ConcurrentMap<DependencyNotation, String> = concurrentMapOf()
        val allVersionsCache: ConcurrentMap<DependencyNotation, List<String>> = concurrentMapOf()
    }


    private fun Collection<String>.toSortedVersions(): List<String> {
        val versions = mapNotNullTo(mutableListOf(), Version::parseOrNull)
        if (size == versions.size) {
            return versions.sortedDescending().map(Version::toString)
        } else {
            return this.toList()
        }
    }

    private val Throwable.isFullyIgnorable: Boolean
        get() = findAll(Throwable::class.java).any {
            it.javaClass.simpleName == getClassSimpleName(ModuleVersionNotFoundException::class.java)
        }

    private val Throwable.isIgnorable: Boolean
        get() = isFullyIgnorable || findAll(Throwable::class.java).any {
            false
        }

}
