package name.remal.gradle_plugins.plugins.generate_sources

import name.remal.createParentDirectories
import name.remal.default
import name.remal.forceDeleteRecursively
import name.remal.gradle_plugins.dsl.BuildTask
import name.remal.gradle_plugins.dsl.extensions.generatedSourcesDir
import name.remal.gradle_plugins.dsl.extensions.noSourceIf
import name.remal.gradle_plugins.dsl.extensions.parents
import name.remal.gradle_plugins.dsl.extensions.requirePlugin
import name.remal.gradle_plugins.plugins.java.JavaAnyPluginId
import name.remal.nullIfEmpty
import org.gradle.api.DefaultTask
import org.gradle.api.Project
import org.gradle.api.file.FileCollection
import org.gradle.api.tasks.*
import java.io.File
import java.nio.charset.StandardCharsets.UTF_8
import kotlin.collections.set

@BuildTask
@CacheableTask
abstract class BaseGenerateTask : DefaultTask() {

    @Input
    var charset: String = UTF_8.name()

    @Classpath
    @InputFiles
    var classpath: FileCollection = project.files()

    @OutputDirectory
    var outputDir: File = project.generatedSourcesDir.resolve(name)

    private final val generateActions: MutableMap<String, (File) -> Unit> = sortedMapOf()

    init {
        requirePlugin(JavaAnyPluginId)
        requirePlugin(GenerateSourcesPlugin::class.java)

        noSourceIf { generateActions.isEmpty() }
    }

    @TaskAction
    protected final fun executeGenerateActions() {
        outputDir.forceDeleteRecursively()
        generateActions.forEach { relativePath, generateAction ->
            val targetFile = File(outputDir, relativePath).absoluteFile
            targetFile.createParentDirectories()
            generateAction(targetFile)
        }
        didWork = true
    }

    protected final fun addGenerateAction(relativePath: String, generateAction: (file: File) -> Unit) {
        if (relativePath in generateActions) logger.warn("Redefining generation of {}", relativePath)
        generateActions[relativePath] = generateAction
    }

    @Input
    protected val objectsToCacheBy = mutableSetOf<Any>().apply {
        add(generateActions.keys)
    }

    @InputFiles
    @Optional
    protected val filesToCacheBy = mutableSetOf<File>().apply {
        var project = project
        while (true) {
            add(project.buildFile)
            project = project.parent ?: break
        }
    }


    @get:Optional
    @get:InputFiles
    @get:PathSensitive(PathSensitivity.RELATIVE)
    private val buildFiles: Set<File> by lazy {
        (sequenceOf(project) + project.parents.asSequence())
            .map(Project::getBuildFile)
            .flatMap {
                sequenceOf(
                    it,
                    it.resolveSibling("settings." + it.extension.nullIfEmpty().default("gradle")),
                    it.resolveSibling("gradle.properties")
                )
            }
            .filter(File::isFile)
            .toSet()
    }

}
