package name.remal.gradle_plugins.plugins.gradle_plugins

import name.remal.concurrentMapOf
import name.remal.gradle_plugins.dsl.Extension
import name.remal.gradle_plugins.dsl.extensions.createWithNotation
import name.remal.gradle_plugins.dsl.extensions.get
import name.remal.gradle_plugins.dsl.extensions.unwrapProviders
import name.remal.gradle_plugins.dsl.utils.DependencyNotation
import name.remal.gradle_plugins.dsl.utils.getGradleLogger
import name.remal.gradle_plugins.plugins.dependencies.DependencyVersionsExtension
import name.remal.nullIfEmpty
import org.gradle.api.Project
import org.gradle.api.artifacts.ConfigurationContainer
import org.gradle.api.artifacts.dsl.DependencyHandler
import org.gradle.api.file.FileCollection
import org.gradle.util.GradleVersion
import java.io.File
import java.util.concurrent.ConcurrentMap

@Extension
class CrossGradleVersionsChecksExtension(
    private val dependencies: DependencyHandler,
    private val configurations: ConfigurationContainer,
    private val project: Project
) {

    companion object {
        private val logger = getGradleLogger(CrossGradleVersionsChecksExtension::class.java)
        private val libraryMarkerClasspathItemsCache: ConcurrentMap<CrossVersionGradleLibrary, Collection<ClasspathItem>> = concurrentMapOf()
        private val notationFilesCache: ConcurrentMap<DependencyNotation, Collection<File>> = concurrentMapOf()
        private val libraryVersionsCache: ConcurrentMap<CrossVersionGradleLibrary, List<GradleVersion>> = concurrentMapOf()
    }

    private val dependencyVersions = project[DependencyVersionsExtension::class.java]


    var minGradleVersion: String = getMinGradleVersion(project) ?: GradleVersion.current().version
        set(value) {
            field = GradleVersion.version(value).version
        }


    val availableVersions: List<GradleVersion>
        get() {
            val minGradleVersion = GradleVersion.version(minGradleVersion)
            return CrossVersionGradleLibrary.values().asSequence()
                .flatMap { getLibraryVersions(it).asSequence() }
                .filter { minGradleVersion <= it }
                .distinct()
                .sorted()
                .toList()
        }

    @JvmOverloads
    fun getClasspathItems(classpathFiles: Iterable<File>, version: GradleVersion = GradleVersion.current()): List<ClasspathItem> {
        if (!classpathFiles.iterator().hasNext()) return emptyList()
        val result: MutableList<ClasspathItem> = classpathFiles.asSequence()
            .distinct()
            .map(::FileClasspathItem)
            .toMutableList()

        CrossVersionGradleLibrary.values().asSequence()
            .filter { version in getLibraryVersions(it) }
            .map { it to getLibraryMarkerClasspathItems(it) }
            .filter { it.second.isNotEmpty() }
            .sortedByDescending { it.second.size }
            .forEach forEachLibrary@{ (lib, classpathItems) ->
                val firstFileIndex = classpathItems.asSequence()
                    .map { result.indexOf(it) }
                    .filter { 0 <= it }
                    .min() ?: return@forEachLibrary

                result.removeAll(classpathItems)

                val newClasspathItem = NotationClasspathItem(lib.notation.withVersion(version.version))
                result.add(firstFileIndex, newClasspathItem)
            }

        return result
    }

    fun replaceClasspathItems(fileCollection: FileCollection, version: GradleVersion): FileCollection {
        return project.files(project.provider<Collection<File>> {
            getClasspathItems(fileCollection, version).asSequence()
                .flatMap {
                    when (it) {
                        is FileClasspathItem -> sequenceOf(it.file)
                        is NotationClasspathItem -> getNotationFiles(it.notation).asSequence()
                    }
                }
                .toList()
        })
    }


    private fun getLibraryVersions(library: CrossVersionGradleLibrary): List<GradleVersion> {
        return libraryVersionsCache.computeIfAbsent(library) {
            dependencyVersions.resolveAllVersions(it.notation.toString()).mapNotNull(GradleVersion::version)
        }
    }

    private fun getLibraryMarkerClasspathItems(library: CrossVersionGradleLibrary): Collection<ClasspathItem> {
        return libraryMarkerClasspathItemsCache.computeIfAbsent(library, {
            configurations.detachedConfiguration(it.dependencyFactory(dependencies)).files.map(::FileClasspathItem)
        })
    }

    private fun getNotationFiles(dependencyNotation: DependencyNotation): Collection<File> {
        return notationFilesCache.computeIfAbsent(dependencyNotation) {
            val dependency = dependencies.createWithNotation(it)
            configurations.detachedConfiguration(dependency).files
        }
    }


    private fun getMinGradleVersion(project: Project): String? {
        val propName = "gradle.min-version"
        val propValue = project.findProperty(propName).unwrapProviders()?.toString().nullIfEmpty() ?: return null
        try {
            GradleVersion.version(propValue)
        } catch (ignored: Exception) {
            logger.warn("Value of property '{}' is not a valid Gradle version: {}", propName, propValue)
            return null
        }
        return propValue
    }

}
