package name.remal.gradle_plugins.plugins.java

import name.remal.createDirectories
import name.remal.default
import name.remal.gradle_plugins.dsl.*
import name.remal.gradle_plugins.dsl.GradleEnumVersion.GRADLE_VERSION_4_3
import name.remal.gradle_plugins.dsl.extensions.*
import name.remal.gradle_plugins.plugins.common.CommonSettingsPlugin
import name.remal.gradle_plugins.plugins.ide.idea.IDEASettings
import name.remal.gradle_plugins.plugins.ide.idea.IdeaSettingsPlugin
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ConfigurationContainer
import org.gradle.api.plugins.ExtensionContainer
import org.gradle.api.tasks.SourceSetContainer
import org.gradle.api.tasks.TaskContainer
import org.gradle.api.tasks.compile.JavaCompile
import org.gradle.util.GradleVersion

private const val APT_CONFIGURATION_NAME = "apt"
private const val APT_OPTIONS_EXTENSION_NAME = "aptOptions"

@Plugin(
    id = "name.remal.apt",
    description = "Plugin that makes easier to use Java annotation processors.",
    tags = ["java", "apt", "annotation-processing"]
)
@ApplyPlugins(JavaAnyPluginId::class)
@ApplyPluginClasses(CommonSettingsPlugin::class)
class AptPlugin : BaseReflectiveProjectPlugin() {

    @CreateExtensionsPluginAction
    fun ExtensionContainer.`Create 'aptOptions' extension`(): AptOptions = create(APT_OPTIONS_EXTENSION_NAME, AptOptions::class.java)

    @CreateConfigurationsPluginAction
    fun ConfigurationContainer.`Create 'apt' configuration`(sourceSets: SourceSetContainer) {
        create(APT_CONFIGURATION_NAME) { conf ->
            conf.description = "Java annotation-processing dependencies"
            sourceSets.all { sourceSet ->
                if (GradleVersion.current() < GradleVersion.version("4.6")) {
                    this[sourceSet.compileOnlyConfigurationName].extendsFrom(conf)
                } else {
                    this[sourceSet.annotationProcessorConfigurationName].extendsFrom(conf)
                }
            }
        }
    }

    @PluginAction
    fun SourceSetContainer.`Setup a dir for generated sources by annotation-processors for 'compileJava' task for all SourceSet`(tasks: TaskContainer) {
        all { sourceSet ->
            tasks.all(JavaCompile::class.java, sourceSet.compileJavaTaskName) { task ->
                val generatedSourcesDir = task.project.generatedSourcesDir.resolve("java-apt/${sourceSet.name}").createDirectories().absoluteFile
                if (GradleVersion.current() < GRADLE_VERSION_4_3) {
                    task.options.compilerArgs.addAll(listOf("-s", generatedSourcesDir.path))
                } else {
                    task.options.annotationProcessorGeneratedSourcesDirectory = generatedSourcesDir
                }
                sourceSet.allJava.srcDir(generatedSourcesDir)

                task.doFirst { generatedSourcesDir.createDirectories() }
            }
        }
    }

    @PluginAction
    fun TaskContainer.`Setup annotation-processor args for all 'compileJava' tasks`() {
        all(JavaCompile::class.java) { task ->
            task.doSetup {
                val aptOptions = task.project.extensions[AptOptions::class.java]
                aptOptions.processorArgs.forEach { name, value ->
                    task.options.compilerArgs.add("-A$name=${value?.toString().default()}")
                }
            }
        }
    }

    @PluginAction
    @WithPluginClasses(IdeaSettingsPlugin::class)
    fun ExtensionContainer.`Configure IDEA`() {
        this[IDEASettings::class.java].addDefaultAnnotationProcessingProfile()
    }

}


val ConfigurationContainer.apt: Configuration get() = this[APT_CONFIGURATION_NAME]


@Extension("Java annotation-processing settings")
class AptOptions {

    @ExtensionProperty("Annotation-processors arguments")
    var processorArgs: MutableMap<String, Any?> = LinkedHashMap()
        set(value) {
            field = LinkedHashMap(value)
        }

}
