package name.remal.gradle_plugins.plugins.java

import name.remal.gradle_plugins.api.AutoService
import name.remal.gradle_plugins.dsl.*
import name.remal.gradle_plugins.dsl.extensions.*
import name.remal.gradle_plugins.plugins.classes_processing.ClassesProcessingPlugin
import name.remal.gradle_plugins.plugins.code_quality.findbugs.FindBugsSettingsPlugin
import name.remal.gradle_plugins.plugins.code_quality.jacoco.JacocoSettingsPlugin
import name.remal.gradle_plugins.plugins.code_quality.spotbugs.SPOTBUGS_PLUGIN_ID
import name.remal.gradle_plugins.plugins.code_quality.spotbugs.SpotBugsPluginId
import name.remal.gradle_plugins.plugins.code_quality.spotbugs.SpotBugsSettingsPlugin
import name.remal.gradle_plugins.plugins.common.CommonSettingsPlugin
import name.remal.gradle_plugins.plugins.dependencies.TransitiveDependenciesConfigurationMatcher
import name.remal.gradle_plugins.plugins.dependencies.TransitiveDependenciesPlugin
import name.remal.gradle_plugins.plugins.merge_resources.MergeResourcesPlugin
import name.remal.gradle_plugins.plugins.testing.TestSettingsPlugin
import org.gradle.api.Project
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ConfigurationContainer
import org.gradle.api.artifacts.ModuleDependency
import org.gradle.api.artifacts.dsl.RepositoryHandler
import org.gradle.api.plugins.JavaPluginConvention
import org.gradle.api.tasks.JavaExec
import org.gradle.api.tasks.SourceSet.MAIN_SOURCE_SET_NAME
import org.gradle.api.tasks.SourceSetContainer
import org.gradle.api.tasks.TaskContainer
import org.gradle.api.tasks.compile.AbstractCompile
import org.gradle.api.tasks.compile.JavaCompile
import org.gradle.api.tasks.testing.Test
import org.gradle.jvm.tasks.Jar
import java.nio.charset.StandardCharsets.UTF_8

const val COMPILE_ONLY_ALL_CONFIGURATION_NAME = "compileOnlyAll"
const val COMPILE_OPTIONAL_CONFIGURATION_NAME = "compileOptional"
const val COMPILE_OPTIONAL_TRANSITIVE_CONFIGURATION_NAME = "compileOptional-transitive"

@Plugin(
    id = "name.remal.java-settings",
    description = "Plugin that configures 'java' plugin if it's applied.",
    tags = ["java"]
)
@WithPlugins(JavaAnyPluginId::class)
@ApplyPluginClasses(CommonSettingsPlugin::class)
@ApplyPluginClassesAtTheEnd(
    TransitiveDependenciesPlugin::class,
    AptPlugin::class,
    ClassesProcessingPlugin::class,
    MergeResourcesPlugin::class,
    TestSettingsPlugin::class,
    JacocoSettingsPlugin::class,
    FindBugsSettingsPlugin::class
)
@SimpleTestAdditionalGradleScript("""
    if (GradleVersion.current() >= GradleVersion.version('5.0')) {
        println 'Applying: $SPOTBUGS_PLUGIN_ID'
        apply plugin: '$SPOTBUGS_PLUGIN_ID'
    }
""")
class JavaSettingsPlugin : BaseReflectiveProjectPlugin() {

    @LowestPriorityPluginAction
    fun Project.`Apply spotbugs-settings plugin`() {
        withPlugin(SpotBugsPluginId) { _ ->
            applyPlugin(SpotBugsSettingsPlugin::class.java)
        }
    }

    @CreateConfigurationsPluginAction
    fun ConfigurationContainer.`Create compileOnlyAll configuration`(sourceSets: SourceSetContainer) {
        create(COMPILE_ONLY_ALL_CONFIGURATION_NAME) { conf ->
            conf.description = "All compileOnly configurations extend this configuration"
            sourceSets.all { sourceSet -> this[sourceSet.compileOnlyConfigurationName].extendsFrom(conf) }
        }
    }

    @CreateConfigurationsPluginAction(""
        + "Create compileOptional configuration.\n"
        + "CompileOnly configuration extends it. Also all *compile configurations extend it.\n"
        + "Dependencies from compileOptional configuration are added to all Test and JavaExec tasks including transitive dependencies. Transitive dependencies can be configured using name.remal.transitive-dependencies plugin."
    )
    fun ConfigurationContainer.createCompileOptionalConfiguration(sourceSets: SourceSetContainer, tasks: TaskContainer) {
        val compileOptional = create(COMPILE_OPTIONAL_CONFIGURATION_NAME) { conf ->
            conf.description = "Optional compile dependencies"

            sourceSets.all(MAIN_SOURCE_SET_NAME) {
                findByName(it.compileOnlyConfigurationName)?.extendsFrom(conf)
            }
            sourceSets.matching { MAIN_SOURCE_SET_NAME != it.name }.all {
                findByName(it.compileConfigurationName)?.extendsFrom(conf)
            }
        }

        val transitiveConf = createDependencyTransformConfiguration(compileOptional, COMPILE_OPTIONAL_TRANSITIVE_CONFIGURATION_NAME) { dep ->
            dep.copy().also { copyDep ->
                if (copyDep is ModuleDependency) {
                    copyDep.isTransitive = true
                }
            }
        }
        transitiveConf.isVisible = false
        transitiveConf.isCanBeConsumed = false
        tasks.all(Test::class.java) { it.doSetup(Int.MIN_VALUE) { it.classpath += transitiveConf } }
        tasks.all(JavaExec::class.java) { it.doSetup(Int.MIN_VALUE) { it.classpath += transitiveConf } }
    }

    @PluginAction
    fun RepositoryHandler.`Use mavenLocal, jcenter and mavenCentral repositories by default`() {
        useDefault {
            jcenter()
            mavenCentral()
            mavenLocal()
        }
    }

    @PluginActionsGroup
    inner class `For all JavaCompile tasks` {

        @PluginAction
        fun TaskContainer.`Set default encoding to UTF-8`() {
            all(JavaCompile::class.java) {
                if (null == it.options.encoding) {
                    it.logDebug("Setting encoding to UTF-8")
                    it.options.encoding = UTF_8.name()
                }
            }
        }

        @PluginAction
        fun TaskContainer.`Enable displaying deprecation warnings`() {
            all(JavaCompile::class.java) {
                it.options.isDeprecation = true
            }
        }

        @PluginAction
        fun TaskContainer.`Add '-parameters' compiler option if targetting Java 8 and above`() {
            all(JavaCompile::class.java) {
                it.doSetupIfAndAfterEvaluate(AbstractCompile::isJava8Compatible) { task ->
                    task.options.compilerArgs.let { compilerArgs ->
                        if ("-parameters" !in compilerArgs) {
                            it.logDebug("Adding -parameters")
                            compilerArgs.add("-parameters")
                        }
                    }
                }
            }
        }

        @PluginAction
        fun TaskContainer.`Add '--module-path' compiler option if sources are compatible with Java 9 and above`() {
            all(JavaCompile::class.java) {
                it.doSetupIf(Int.MAX_VALUE, { it.sourceJavaVersion.isJava9Compatible }) { task ->
                    task.options.compilerArgs.let { compilerArgs ->
                        if ("--module-path" !in compilerArgs) {
                            var classpath = task.classpath
                            task.options.annotationProcessorPath?.let { classpath += it }

                            val modulePath = classpath.asPath
                            if (modulePath.isNotEmpty()) {
                                it.logDebug("Adding --module-path {}", modulePath)
                                compilerArgs.add("--module-path")
                                compilerArgs.add(modulePath)
                            }
                        }
                    }
                }
            }
        }

    }

    @PluginAction
    fun TaskContainer.`Add Automatic-Module-Name manifest attribute in result jar archive if targetting Java 8 or below`(javaConvention: JavaPluginConvention) {
        all(Jar::class.java) {
            it.doSetup {
                val targetCompatibility = javaConvention.targetCompatibility
                if (targetCompatibility.isJava9Compatible) return@doSetup

                val moduleNameAttribute = "Automatic-Module-Name"
                if (it.classifier in setOf("sources", "javadoc", "groovydoc", "kotlindoc", "dokka", "kdoc", "scaladoc")) {
                    it.logDebug("Skip adding {} manifest attribute for classifier '{}'", moduleNameAttribute, it.classifier)
                    return@doSetup
                }
                val attributes = it.manifest.attributes
                if (moduleNameAttribute !in attributes) {
                    val moduleName = it.project.javaModuleName
                    it.logDebug("Adding {} manifest attribute - {}", moduleNameAttribute, moduleName)
                    it.inputs[JavaSettingsPlugin::class.java.name + ':' + moduleNameAttribute] = moduleName
                    attributes[moduleNameAttribute] = moduleName
                }
            }
        }
    }

}


val ConfigurationContainer.compileOnlyAll: Configuration get() = this[COMPILE_ONLY_ALL_CONFIGURATION_NAME]
val ConfigurationContainer.compileOptional: Configuration get() = this[COMPILE_OPTIONAL_CONFIGURATION_NAME]
val ConfigurationContainer.compileOptionalTransitive: Configuration get() = this[COMPILE_OPTIONAL_TRANSITIVE_CONFIGURATION_NAME]


@AutoService
class JavaSettingsPluginTransitiveDependenciesConfigurationMatcher : TransitiveDependenciesConfigurationMatcher {
    override fun matches(project: Project, configuration: Configuration): Boolean {
        if (!project.isPluginApplied(JavaSettingsPlugin::class.java)) return false
        if (COMPILE_ONLY_ALL_CONFIGURATION_NAME == configuration.name) return true
        if (COMPILE_OPTIONAL_CONFIGURATION_NAME == configuration.name) return true
        if (COMPILE_OPTIONAL_TRANSITIVE_CONFIGURATION_NAME == configuration.name) return true
        return false
    }
}
