package name.remal.gradle_plugins.plugins.java

import name.remal.concurrentMapOf
import name.remal.gradle_plugins.dsl.BaseReflectiveProjectPlugin
import name.remal.gradle_plugins.dsl.Plugin
import name.remal.gradle_plugins.dsl.PluginAction
import name.remal.gradle_plugins.dsl.WithPlugins
import name.remal.gradle_plugins.dsl.extensions.beforeResolve
import name.remal.gradle_plugins.dsl.extensions.java
import name.remal.gradle_plugins.dsl.extensions.jcenterIfNotAdded
import org.gradle.api.Project
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ConfigurationContainer
import org.gradle.api.artifacts.ResolvedArtifact
import org.gradle.api.artifacts.component.ModuleComponentIdentifier
import org.gradle.api.artifacts.component.ModuleComponentSelector
import org.gradle.api.artifacts.dsl.RepositoryHandler

@Plugin(
    id = "name.remal.experimental-jsr205-jsr305-split-package-fixer",
    description = "Plugin that fixes split package situation of JSR 205 and JSR 305 artifacts.",
    tags = ["dependencies", "java9", "jigsaw", "split-package", "jsr205", "jsr305"]
)
@WithPlugins(JavaAnyPluginId::class)
class Jsr205Jsr305SplitPackageFixerPlugin : BaseReflectiveProjectPlugin() {

    companion object {
        internal const val MERGED_ARTIFACT_GROUP = "name.remal.merged"
        internal const val MERGED_ARTIFACT_ID = "jsr205-jsr305"
        internal const val MERGED_ARTIFACT_ID_REVERSE = "jsr305-jsr205"
    }

    private val versionsInfoCache = concurrentMapOf<Configuration, VersionsInfo>()

    @PluginAction(isHidden = true)
    fun RepositoryHandler.addJCenter() {
        jcenterIfNotAdded()
    }

    @PluginAction(isHidden = true)
    fun ConfigurationContainer.processConfigurations(project: Project) {
        all {
            it.beforeResolve { configuration ->
                if (!configuration.isCanBeResolved) {
                    return@beforeResolve
                }

                if (!project.java.sourceCompatibility.isJava9Compatible) {
                    return@beforeResolve
                }

                val versionsInfo = VersionsInfo()
                fillVersionsInfo(configuration, versionsInfo)
                versionsInfoCache[configuration] = versionsInfo

                val substitutionNotation: String? = with(versionsInfo) {
                    if (jsr205Version != null && jsr305Version != null) {
                        "$MERGED_ARTIFACT_GROUP:$MERGED_ARTIFACT_ID:$jsr205VersionPrefix$jsr205Version-$jsr305VersionPrefix$jsr305Version"
                    } else if (jsr205Version != null) {
                        "$MERGED_ARTIFACT_GROUP:$MERGED_ARTIFACT_ID:$jsr205VersionPrefix$jsr205Version-+"
                    } else if (jsr305Version != null) {
                        "$MERGED_ARTIFACT_GROUP:$MERGED_ARTIFACT_ID_REVERSE:$jsr305VersionPrefix$jsr305Version-+"
                    } else {
                        null
                    }
                }

                if (substitutionNotation != null) {
                    val notationsToSubstitute = versionsInfo.notationsToSubstitute
                    configuration.resolutionStrategy {
                        it.dependencySubstitution {
                            it.all substitution@{ subst ->
                                val requested = subst.requested as? ModuleComponentSelector ?: return@substitution
                                if ("${requested.group}:${requested.module}" in notationsToSubstitute) {
                                    subst.useTarget(substitutionNotation, "Using ${notationsToSubstitute.joinToString(", ")} at the same time causes split package compile error, thus merged dependency is used.")
                                }
                            }
                        }
                    }
                }
            }
        }
    }


    private fun ConfigurationContainer.fillVersionsInfo(configuration: Configuration, versionsInfo: VersionsInfo) {
        forEach {
            if (it.isCanBeResolved && configuration in it.extendsFrom) {
                fillVersionsInfo(it, versionsInfo)
                if (versionsInfo.isFilled) {
                    return
                }
            }
        }

        versionsInfoCache[configuration]?.let { cachedVersionInfo ->
            if (cachedVersionInfo.jsr205Version != null) {
                versionsInfo.jsr205VersionPrefix = cachedVersionInfo.jsr205VersionPrefix
                versionsInfo.jsr205Version = cachedVersionInfo.jsr205Version
            }
            if (cachedVersionInfo.jsr305Version != null) {
                versionsInfo.jsr305VersionPrefix = cachedVersionInfo.jsr305VersionPrefix
                versionsInfo.jsr305Version = cachedVersionInfo.jsr305Version
            }
            versionsInfo.notationsToSubstitute.addAll(cachedVersionInfo.notationsToSubstitute)
            return
        }

        with(versionsInfo) {
            val resolvedArtifacts: Iterable<ResolvedArtifact> = configuration.copyRecursive().resolvedConfiguration.resolvedArtifacts
            resolvedArtifacts.forEach { resolvedArtifact ->
                val id = resolvedArtifact.id.componentIdentifier as? ModuleComponentIdentifier ?: return@forEach

                if (jsr205Version == null) {
                    if (id.group == "javax.annotation" && id.module == "javax.annotation-api") {
                        notationsToSubstitute.add("${id.group}:${id.module}")
                        jsr205VersionPrefix = "javax_"
                        jsr205Version = id.version

                    } else if (id.group == "jakarta.annotation" && id.module == "jakarta.annotation-api") {
                        notationsToSubstitute.add("${id.group}:${id.module}")
                        jsr205VersionPrefix = "jakarta_"
                        jsr205Version = id.version
                    }
                }

                if (jsr305Version == null) {
                    if (id.group == "com.google.code.findbugs" && id.module == "jsr305") {
                        notationsToSubstitute.add("${id.group}:${id.module}")
                        jsr305VersionPrefix = "findbugs_"
                        jsr305Version = id.version
                    }
                }
            }
        }
    }

    private data class VersionsInfo(
        val notationsToSubstitute: MutableSet<String> = hashSetOf(),
        var jsr205Version: String? = null,
        var jsr205VersionPrefix: String = "",
        var jsr305Version: String? = null,
        var jsr305VersionPrefix: String = ""
    ) {

        val isFilled: Boolean get() = jsr205Version != null && jsr305Version != null

    }

}
