package name.remal.gradle_plugins.plugins.kotlin

import name.remal.KotlinAllOpen
import name.remal.debug
import name.remal.gradle_plugins.api.BuildTimeConstants.getClassName
import name.remal.gradle_plugins.dsl.*
import name.remal.gradle_plugins.dsl.extensions.*
import name.remal.gradle_plugins.plugins.common.CommonSettingsPlugin
import name.remal.gradle_plugins.plugins.java.JavaSettingsPlugin
import name.remal.isClassExists
import name.remal.version.Version
import org.gradle.api.plugins.ExtensionContainer
import org.gradle.api.plugins.PluginContainer
import org.gradle.api.plugins.UnknownPluginException
import org.gradle.api.tasks.SourceSetContainer
import org.gradle.api.tasks.TaskContainer
import org.gradle.api.tasks.compile.AbstractCompile
import org.jetbrains.kotlin.allopen.gradle.AllOpenExtension
import org.jetbrains.kotlin.gradle.plugin.KaptExtension
import org.jetbrains.kotlin.gradle.plugin.KotlinPluginWrapper
import org.jetbrains.kotlin.gradle.tasks.KotlinCompile

@Plugin(
    id = "name.remal.kotlin-settings",
    description = "Plugin that configures Kotlin plugins if some of them are applied.",
    tags = ["kotlin"]
)
@WithPlugins(KotlinJvmPluginId::class)
@ApplyPluginClasses(CommonSettingsPlugin::class, JavaSettingsPlugin::class)
@ApplyOptionalPlugins(
    KotlinPluginAllOpenPluginId::class,
    KotlinPluginJpaPluginId::class,
    KotlinPluginNoArgPluginId::class,
    KotlinPluginSpringPluginId::class
)
class KotlinJvmSettingsPlugin : BaseReflectiveProjectPlugin() {

    companion object {
        private val kotlinPluginWrapperClassName = getClassName(KotlinPluginWrapper::class.java)
        private val isKotlinPluginWrapperClassAvailable = isClassExists(kotlinPluginWrapperClassName, KotlinJvmSettingsPlugin::class.java.classLoader)
    }


    @PluginAction(order = Int.MAX_VALUE)
    @AfterProjectEvaluation
    fun SourceSetContainer.`Make source directories unique for all SourceSets`() {
        all {
            arrayOf(it.java, it.allJava, it.resources, it.allSource).forEach {
                it.makeSrcDirsUnique()
            }
        }
    }

    @PluginActionsGroup
    inner class `For all KotlinCompile tasks` {

        @PluginAction("If targetCompatibility is Java 8 compatible, Set jvmTarget = '1.8', javaParameters = true and add -Xjvm-default=compatibility compiler flag")
        fun TaskContainer.setJvmTarget8(plugins: PluginContainer) {
            all(KotlinCompile::class.java) {
                it.doSetupIfAndAfterEvaluate(AbstractCompile::isJava8Compatible) { task ->
                    task.kotlinOptions.run {
                        jvmTarget = "1.8"
                        javaParameters = true

                        val isJvmDefaultSupported: Boolean = if (!isKotlinPluginWrapperClassAvailable) {
                            true
                        } else {
                            (plugins.kotlinPluginVersion ?: Version.create(9999999)) >= Version.create(1, 2, 60)
                        }
                        if (isJvmDefaultSupported && freeCompilerArgs.none { it.startsWith("-Xjvm-default=") }) {
                            freeCompilerArgs += listOf("-Xjvm-default=compatibility")
                        }
                    }
                }
            }
        }

        private val PluginContainer.kotlinPluginVersion: Version?
            get() {
                try {
                    return getPlugin(KotlinPluginWrapper::class.java).kotlinPluginVersion.let(Version::parseOrNull)
                } catch (e: UnknownPluginException) {
                    logger.debug(e)
                    return null
                }
            }

    }

    @WithPlugins(KotlinPluginKaptPluginId::class)
    @PluginAction("Set kapt.correctErrorTypes = true if 'kapt' Kotlin plugin enabled")
    fun ExtensionContainer.`setKaptCorrectErrorTypes`() {
        this[KaptExtension::class.java].correctErrorTypes = true
    }

    @WithPlugins(KotlinPluginAllOpenPluginId::class)
    @PluginActionsGroup(order = Int.MAX_VALUE)
    inner class `If 'all-open' Kotlin plugin is applied, add these annotations as open-annotation` {

        @PluginAction(value = "name.remal.KotlinAllOpen")
        fun ExtensionContainer.name_remal_KotlinAllOpen() {
            this[AllOpenExtension::class.java].annotation(KotlinAllOpen::class.java.name)
        }

        @PluginAction(value = "javax.inject.Inject")
        fun ExtensionContainer.javax_inject_Inject() {
            this[AllOpenExtension::class.java].annotation("javax.inject.Inject")
        }

    }

}
