package name.remal.gradle_plugins.plugins.noarg_constructor

import com.google.common.cache.CacheBuilder
import com.google.common.cache.CacheLoader
import com.google.common.cache.LoadingCache
import name.remal.CLASS_FILE_NAME_SUFFIX
import name.remal.accept
import name.remal.gradle_plugins.api.AutoService
import name.remal.gradle_plugins.api.classes_processing.BytecodeModifier
import name.remal.gradle_plugins.api.classes_processing.ClassesProcessor
import name.remal.gradle_plugins.api.classes_processing.ClassesProcessorsGradleTaskFactory
import name.remal.gradle_plugins.api.classes_processing.ProcessContext
import name.remal.gradle_plugins.dsl.extensions.isPluginApplied
import org.gradle.api.tasks.compile.AbstractCompile
import org.objectweb.asm.ClassReader
import org.objectweb.asm.ClassWriter
import org.objectweb.asm.Opcodes.*
import org.objectweb.asm.tree.*
import java.util.concurrent.atomic.AtomicReference

class NoargClassesProcessor : ClassesProcessor {

    private val contextRef = AtomicReference<ProcessContext>()

    private val hasPublicNoargCache: LoadingCache<String, Boolean> = CacheBuilder.newBuilder()
        .build<String, Boolean>(object : CacheLoader<String, Boolean>() {
            override fun load(relativePath: String): Boolean {
                if ("java/lang/Object.class" == relativePath) return true
                val context = contextRef.get()
                val bytecode = context.readBinaryResource(relativePath)
                    ?: context.readClasspathBinaryResource(relativePath)
                    ?: return false
                val classNode = ClassNode().also { ClassReader(bytecode).accept(it) }
                return classNode.hasPublicOrProtectedNoargConstructor
            }
        })

    override fun process(bytecode: ByteArray, bytecodeModifier: BytecodeModifier, className: String, resourceName: String, context: ProcessContext) {
        contextRef.compareAndSet(null, context)

        val classReader = ClassReader(bytecode)
        val classNode = ClassNode().also { classReader.accept(it) }

        if (0 != (ACC_INTERFACE and classNode.access)) return
        if (0 != (ACC_ANNOTATION and classNode.access)) return
        if (0 != (ACC_ENUM and classNode.access)) return
        if (0 != (ACC_MODULE and classNode.access)) return

        if (null != classNode.outerClass && 0 == (ACC_STATIC and classNode.access)) return

        if (classNode.hasNoargConstructor) {
            hasPublicNoargCache.put(resourceName, true)
            return
        }

        if (!hasPublicNoargCache[classNode.superName + CLASS_FILE_NAME_SUFFIX]) {
            hasPublicNoargCache.put(resourceName, false)
            return
        }

        hasPublicNoargCache.put(resourceName, true)

        if (null == classNode.methods) classNode.methods = mutableListOf()
        classNode.methods.add(MethodNode(ACC_PROTECTED or ACC_SYNTHETIC, "<init>", "()V", null, null).also { method ->
            method.invisibleAnnotations = mutableListOf()
            method.invisibleAnnotations.add(AnnotationNode("Ledu/umd/cs/findbugs/annotations/SuppressFBWarnings;"))

            method.instructions = InsnList().also {
                it.add(LabelNode())
                it.add(VarInsnNode(ALOAD, 0))
                it.add(MethodInsnNode(INVOKESPECIAL, classNode.superName, method.name, method.desc, false))
                it.add(InsnNode(RETURN))
            }
            method.maxLocals = 1
            method.maxStack = 1
        })

        val classWriter = ClassWriter(classReader, 0)
        classNode.accept(classWriter)
        bytecodeModifier.modify(classWriter.toByteArray())
    }


    private val ClassNode.noargConstructor get() = methods?.firstOrNull { it.name == "<init>" && it.desc == "()V" }
    private val ClassNode.hasNoargConstructor get() = noargConstructor != null
    private val ClassNode.hasPublicOrProtectedNoargConstructor get() = noargConstructor.let { it != null && ((ACC_PUBLIC or ACC_PROTECTED) and it.access) != 0 }

}


@AutoService
class NoargClassesProcessorFactory : ClassesProcessorsGradleTaskFactory {
    override fun createClassesProcessors(compileTask: AbstractCompile): List<ClassesProcessor> {
        if (!compileTask.project.isPluginApplied(NoargConstructorPlugin::class.java)) return emptyList()
        return listOf(NoargClassesProcessor())
    }
}
