package name.remal.gradle_plugins.plugins.publish.nexus_staging

import name.remal.default
import name.remal.gradle_plugins.dsl.BuildTask
import name.remal.gradle_plugins.dsl.extensions.*
import name.remal.gradle_plugins.dsl.utils.retryIO
import name.remal.gradle_plugins.utils.*
import name.remal.queueOf
import org.gradle.api.DefaultTask
import org.gradle.api.artifacts.repositories.MavenArtifactRepository
import org.gradle.api.artifacts.repositories.PasswordCredentials
import org.gradle.api.publish.maven.MavenPublication
import org.gradle.api.tasks.TaskAction
import java.net.URI
import java.time.ZonedDateTime
import java.util.*

@BuildTask
class ReleaseNexusStagingRepository : DefaultTask() {

    companion object {
        private val SLEEP_MILLIS_DURATION: Long = 2500
    }


    init {
        requirePlugin(MavenPublishNexusStagingPlugin::class.java)
    }


    lateinit var publication: MavenPublication

    lateinit var repository: MavenArtifactRepository

    lateinit var stagingNexusURI: URI

    var stagingProfileId: String? = null


    private val nexusStagingApi: NexusStagingApi by lazy {
        val passwordCredentials = repository.getCredentials(PasswordCredentials::class.java)
        return@lazy newJsonRetrofitBuilder {
            it.addInterceptor {
                it.proceed(it.request().newBuilder()
                    .addAuthorization(passwordCredentials.username.default(), passwordCredentials.password.default())
                    .build())
            }
        }
            .baseUrl(stagingNexusURI)
            .create(NexusStagingApi::class.java)
    }


    @TaskAction
    protected fun release() {
        synchronized(ReleaseNexusStagingRepository::class.java) {
            doRelease()
        }
    }


    private fun doRelease() {
        val stagingNexusURI: URI = this.stagingNexusURI

        val stagingProfileId: String = this.stagingProfileId ?: run {
            val profiles = retryIO { nexusStagingApi.getProfiles() }.data
                .filter { it.repositoryType == "maven2" }
                .sortedByDescending(NexusStagingProfile::name)
            val groupId = publication.groupId
            return@run profiles.firstOrNull { (groupId + '.').startsWith(it.name + '.') }?.id
                ?.also { logLifecycle("Staging profile: {}", it) }
                ?: throw IllegalStateException("$stagingNexusURI: There is not matched staging profile for groupId '$groupId': ${profiles.map(NexusStagingProfile::name)}")
        }


        val repositoriesToRelease = retryIO { nexusStagingApi.getProfileRepositories(stagingProfileId) }.data.asSequence()
            .filter { it.type != "dropped" }
            .toList()

        if (repositoriesToRelease.isNotEmpty()) {
            repositoriesToRelease.forEach(this::doRelease)
        } else {
            logLifecycle("There are no staging repositories to release with this staging profile.")
        }

        didWork = true
    }

    private tailrec fun doRelease(repository: NexusStagingRepository) {
        if (repository.transitioning) {
            logDebug("{} - in transitioning state, waiting...", repository.repositoryURI)
            Thread.sleep(SLEEP_MILLIS_DURATION)
            return doRelease(fetchRepository(repository.repositoryId) ?: return)
        }

        val startTimestamp = ZonedDateTime.now()
        val loggedEvents = mutableSetOf<NexusStagingActionEvent>()
        fun waitForActionToComplete() {
            var isSuccess = true
            while (true) {
                Thread.sleep(SLEEP_MILLIS_DURATION)
                val actions = fetchRepositoryActivity(repository.repositoryId)
                    .asSequence()
                    .filter { it.started >= startTimestamp }
                    .toList()
                if (actions.isEmpty()) {
                    break
                }

                actions.asSequence()
                    .flatMap { action ->
                        action.events.asSequence()
                            .filter { it.timestamp >= startTimestamp }
                            .map { NexusStagingActionEvent(action.name, action.started, it) }
                    }
                    .filter(loggedEvents::add)
                    .map(NexusStagingActionEvent::event)
                    .forEach { event ->
                        if (event.severity == 0) {
                            event.format().forEach { logLifecycle("    {}", it) }
                        } else {
                            event.format().forEach { logError("    {}", it) }
                            isSuccess = false
                        }
                    }

                if (actions.all { it.stopped != null }) {
                    break
                }
            }

            if (!isSuccess) {
                throw NexusRepositoryTransitionException(repository, loggedEvents)
            }
        }

        fun waitForActionToCompleteOrDrop() {
            try {
                waitForActionToComplete()

            } catch (exception: NexusRepositoryTransitionException) {
                logError("{} - dropping", repository.repositoryURI)
                try {
                    retryIO { nexusStagingApi.bulkAction("drop", repository.repositoryId).send() }
                } catch (e: Exception) {
                    e.addSuppressed(exception)
                    throw e
                }
                throw exception
            }
        }


        if (repository.type == "open") {
            logLifecycle("{} - closing", repository.repositoryURI)
            retryIO { nexusStagingApi.bulkAction("close", repository.repositoryId).send() }
            waitForActionToCompleteOrDrop()
            return doRelease(fetchRepository(repository.repositoryId) ?: return)

        } else if (repository.type == "closed") {
            run {
                val content = try {
                    fetchRepositoryContent(repository.repositoryId)
                } catch (e: Throwable) {
                    logWarn(e)
                    return@run
                }
                if (content.isEmpty()) {
                    logLifecycle("{} - is empty", repository.repositoryURI)
                } else {
                    logLifecycle("{} - content:{}", repository.repositoryURI, buildString {
                        content.forEach { append("\n    ").append(it) }
                    })
                }
            }

            logLifecycle("{} - releasing", repository.repositoryURI)
            retryIO { nexusStagingApi.bulkAction("promote", repository.repositoryId).send() }
            waitForActionToCompleteOrDrop()
            return
        }
    }


    private fun fetchRepository(repositoryId: String): NexusStagingRepository? = retryIO {
        try {
            nexusStagingApi.getRepository(repositoryId)
        } catch (e: RetrofitCallException) {
            if (e.status == 404) {
                null
            } else {
                throw e
            }
        }
    }

    private fun fetchRepositoryActivity(repositoryId: String): List<NexusStagingRepositoryAction> = retryIO {
        try {
            nexusStagingApi.getRepositoryActivity(repositoryId)
        } catch (e: RetrofitCallException) {
            if (e.status == 404) {
                emptyList()
            } else {
                throw e
            }
        }
    }

    private fun fetchRepositoryContent(repositoryId: String): SortedSet<String> {
        val relativePathsQueue = queueOf<String>()
        relativePathsQueue.add("")

        val processedDirectories = hashSetOf<String>()
        processedDirectories.addAll(relativePathsQueue)

        val relativePaths = sortedSetOf<String>()
        while (true) {
            val currentRelativePath = relativePathsQueue.poll() ?: break
            val contentItems = retryIO {
                nexusStagingApi.getRepositoryContent(repositoryId, currentRelativePath).data
            }

            contentItems.forEach { contentItem ->
                val relativePath = contentItem.relativePath.trimStart('/')
                if (contentItem.leaf) {
                    relativePaths.add(relativePath)
                } else if (processedDirectories.add(relativePath)) {
                    relativePathsQueue.add(relativePath)
                }
            }
        }
        return relativePaths
    }

}
