package name.remal.gradle_plugins.plugins.testing

import name.remal.gradle_plugins.dsl.*
import name.remal.gradle_plugins.dsl.extensions.*
import name.remal.gradle_plugins.plugins.assertj.AssertJGenerate
import name.remal.gradle_plugins.plugins.java.JavaAnyPluginId
import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.artifacts.ConfigurationContainer
import org.gradle.api.tasks.SourceTask
import org.gradle.api.tasks.TaskContainer
import org.gradle.api.tasks.compile.AbstractCompile
import org.gradle.api.tasks.testing.AbstractTestTask
import org.gradle.language.base.plugins.LifecycleBasePlugin.CHECK_TASK_NAME

@Plugin(
    id = "name.remal.disable-tests",
    description = "Plugin that disables all check and test tasks. Also this plugin disables test source sets tasks.",
    tags = ["java"]
)
@WithPlugins(JavaAnyPluginId::class)
@ApplyPluginClasses(TestSourceSetsPlugin::class)
class DisableTestsPlugin : BaseReflectiveProjectPlugin() {

    @HighestPriorityPluginAction
    fun Project.`Display warning message`() {
        "| Tests are disabled for project '$path' |".let { msg ->
            logger.warn("-".repeat(msg.length))
            logger.warn(msg)
            logger.warn("-".repeat(msg.length))
        }
    }

    @PluginAction
    fun TaskContainer.`Skip test tasks`(project: Project, testSourceSets: TestSourceSetContainer) {
        project.setupTasksDependenciesAfterEvaluateOrNow(Int.MAX_VALUE) { _ ->
            val tasksToDisable = mutableListOf<Task>()
            forEach forEachTask@{ task ->
                if (task is AbstractTestTask
                    || task is AssertJGenerate
                ) {
                    tasksToDisable.add(task)
                    return@forEachTask
                }

                if (task.name == CHECK_TASK_NAME) {
                    tasksToDisable.add(task)
                    return@forEachTask
                }

                testSourceSets.forEach { sourceSet ->
                    if (task.name in sourceSet.sourceSetTaskNames) {
                        tasksToDisable.add(task)
                        return@forEachTask
                    }
                    if (task is AbstractCompile && task.isCompilingSourceSet(sourceSet)) {
                        tasksToDisable.add(task)
                        return@forEachTask
                    }
                    if (task is SourceTask && task.isProcessingSourceSet(sourceSet)) {
                        tasksToDisable.add(task)
                        return@forEachTask
                    }
                }
            }

            if (tasksToDisable.none(Task::isRequested)) {
                tasksToDisable.forEach { task ->
                    task.enabled = false
                    task.setDependsOn(emptyList<String>())
                }
            }
        }
    }

    @PluginAction
    fun TestSourceSetContainer.`Clean all test dependencies`(configurations: ConfigurationContainer, project: Project) {
        all { testSourceSet ->
            testSourceSet.compileClasspath = project.emptyFileTree()
            testSourceSet.runtimeClasspath = project.emptyFileTree()
            testSourceSet.sourceSetConfigurationNames.forEach { confName ->
                configurations.all(confName) { conf ->
                    conf.dependencies.clear()
                    conf.setExtendsFrom(emptyList())

                    conf.allDependencies.whenObjectAdded {
                        conf.dependencies.clear()
                        conf.setExtendsFrom(emptyList())
                    }
                }
            }
        }
    }

}
