package name.remal.gradle_plugins.plugins.testing

import name.remal.gradle_plugins.dsl.*
import name.remal.gradle_plugins.dsl.extensions.*
import name.remal.gradle_plugins.plugins.common.CommonSettingsPlugin
import name.remal.gradle_plugins.plugins.ide.idea.IdeaPluginId
import name.remal.gradle_plugins.plugins.java.JavaAnyPluginId
import org.gradle.api.NamedDomainObjectContainer
import org.gradle.api.Project
import org.gradle.api.artifacts.ConfigurationContainer
import org.gradle.api.plugins.ExtensionContainer
import org.gradle.api.plugins.JavaBasePlugin.VERIFICATION_GROUP
import org.gradle.api.plugins.JavaPlugin.TEST_TASK_NAME
import org.gradle.api.tasks.SourceSet
import org.gradle.api.tasks.SourceSet.TEST_SOURCE_SET_NAME
import org.gradle.api.tasks.SourceSetContainer
import org.gradle.api.tasks.TaskContainer
import org.gradle.api.tasks.testing.Test
import org.gradle.plugins.ide.idea.model.IdeaModel
import org.gradle.plugins.ide.idea.model.internal.GeneratedIdeaScope
import java.util.concurrent.atomic.AtomicBoolean

@Plugin(
    id = "name.remal.test-source-sets",
    description = "Plugin that provides testSourceSet object for creating new source sets for testing. For all created source sets Test task is created. All dependencies are inherited from 'test' source set.",
    tags = ["java", "test"]
)
@WithPlugins(JavaAnyPluginId::class)
@ApplyPluginClasses(CommonSettingsPlugin::class, TestSettingsPlugin::class)
@SimpleTestAdditionalGradleScript("testSourceSets { additionalTestSourceSet }")
class TestSourceSetsPlugin : BaseReflectiveProjectPlugin() {

    @CreateExtensionsPluginAction
    fun ExtensionContainer.`Create testSourceSets extension`(sourceSets: SourceSetContainer, project: Project, configurations: ConfigurationContainer) {
        val container: NamedDomainObjectContainer<SourceSet> = project.containerWithFactory(SourceSet::class.java, sourceSets::create)
        container.add(sourceSets.test)
        container.whenObjectAdded { sourceSets.add(it) }
        val isRemovingRunning = AtomicBoolean(false)
        sourceSets.whenObjectRemoved { sourceSet ->
            if (isRemovingRunning.compareAndSet(false, true)) {
                container.remove(sourceSet)
                isRemovingRunning.set(false)
            }
        }
        container.whenObjectRemoved { testSourceSet ->
            if (isRemovingRunning.compareAndSet(false, true)) {
                sourceSets.remove(testSourceSet)
                isRemovingRunning.set(false)
            }
        }

        container.whenObjectAdded { testSourceSet ->
            testSourceSet.compileClasspath = project.files(
                project.provider { sourceSets.main.output },
                project.provider { configurations[testSourceSet.compileClasspathConfigurationName] }
            )
            testSourceSet.runtimeClasspath = project.files(
                testSourceSet.output,
                project.provider { sourceSets.main.output },
                project.provider { configurations[testSourceSet.runtimeClasspathConfigurationName] }
            )
        }

        create(
            TestSourceSetContainer::class.java,
            "testSourceSets",
            TestSourceSetContainer::class.java.getDelegateClassOf(NamedDomainObjectContainer::class.java),
            container
        )
    }

    @CreateConfigurationsPluginAction
    fun TestSourceSetContainer.`Make all testSourceSets configurations extends configurations of 'test' source set`(configurations: ConfigurationContainer) {
        val origTestSourceSet = this.test
        whenObjectAdded {
            it.forEachCorrespondingConfigurationName(origTestSourceSet) { itConfName, testConfName ->
                configurations.all(itConfName) { itConf ->
                    val testConf = configurations.findByName(testConfName) ?: return@all
                    itConf.extendsFrom(testConf)
                }
                configurations.all(testConfName) { testConf ->
                    val itConf = configurations.findByName(itConfName) ?: return@all
                    itConf.extendsFrom(testConf)
                }
            }
        }
    }

    @PluginAction
    fun TestSourceSetContainer.`Create Test task for all testSourceSets`(tasks: TaskContainer) {
        whenObjectAdded { testSourceSet ->
            tasks.create(testSourceSet.testTaskName, Test::class.java) {
                it.group = VERIFICATION_GROUP
                it.description = "Runs ${testSourceSet.name} tests"
                it.conventionMapping.map("testClassesDirs") { testSourceSet.output.classesDirs }
                it.conventionMapping.map("classpath") { testSourceSet.runtimeClasspath }
            }
        }
    }

    @PluginAction(order = Int.MAX_VALUE)
    fun TaskContainer.`Create runAllTests task`(testSourceSets: TestSourceSetContainer) {
        create("runAllTests") { task ->
            task.group = VERIFICATION_GROUP
            task.description = "Run all test tasks for every source-set"
            task.dependsOn {
                testSourceSets.asSequence()
                    .map(SourceSet::testTaskName)
                    .mapNotNull(this::findByName)
                    .toList()
            }
        }
    }

    @PluginAction(order = Int.MAX_VALUE)
    @WithPlugins(IdeaPluginId::class)
    fun ExtensionContainer.`Setup IDEA scopes`(configurations: ConfigurationContainer, testSourceSets: TestSourceSetContainer) {
        this.findByType(IdeaModel::class.java)?.module?.apply {
            testSourceSets.all { testSourceSet ->
                configurations.findByName(testSourceSet.compileClasspathConfigurationName)?.let { addToScope(GeneratedIdeaScope.TEST, it) }
                configurations.findByName(testSourceSet.runtimeClasspathConfigurationName)?.let { addToScope(GeneratedIdeaScope.TEST, it) }
            }
        }
    }

}

val SourceSet.testTaskName: String
    get() {
        if (TEST_SOURCE_SET_NAME == name) return TEST_TASK_NAME
        return TEST_TASK_NAME + name.capitalize()
    }
