package name.remal.gradle_plugins.utils

import name.remal.concurrentMapOf
import name.remal.gradle_plugins.api.BuildTimeConstants.getStringProperties
import name.remal.gradle_plugins.dsl.extensions.applyPlugin
import name.remal.gradle_plugins.dsl.extensions.get
import name.remal.gradle_plugins.dsl.extensions.hasRepositoryWithDynamicVersionsSupport
import name.remal.gradle_plugins.dsl.utils.tryToMakeDynamicPatchVersion
import name.remal.gradle_plugins.plugins.dependencies.DependencyVersionsExtension
import name.remal.gradle_plugins.plugins.dependencies.DependencyVersionsPlugin
import org.gradle.api.Project

private val versionProperties = getStringProperties("*.version").mapKeys { it.key.substring(0, it.key.length - ".version".length) }
private val latestVersionProperties = getStringProperties("*.latest-version").mapKeys { it.key.substring(0, it.key.length - ".latest-version".length) }
private val dependencyNotationProperties = getStringProperties("*.dependency-notation").mapKeys { it.key.substring(0, it.key.length - ".dependency-notation".length) }

/**
 * There is an issue in Gradle 4.4 when getPredefinedDynamicVersionProperty() causes infinite loop.
 * Let's return a latest version property value if this issue occurs.
 */
private val isResolving = concurrentMapOf<String, Boolean>()

internal fun Project.getPredefinedDynamicVersionProperty(id: String): String {
    var version = versionProperties[id]
    if (version != null && !version.contains('+')) {
        return version
    }

    if (!gradle.startParameter.isOffline && repositories.hasRepositoryWithDynamicVersionsSupport) {
        version = version ?: latestVersionProperties[id]
        if (version != null) {
            val dependencyVersions = extensions.findByType(DependencyVersionsExtension::class.java)
                ?: run {
                    applyPlugin(DependencyVersionsPlugin::class.java)
                    extensions[DependencyVersionsExtension::class.java]
                }

            val notation = dependencyNotationProperties[id]
            if (notation != null) {
                if (isResolving[notation] != true) {
                    isResolving[notation] = true
                    try {
                        if (!version.contains('+')) {
                            version = tryToMakeDynamicPatchVersion(version)
                        }
                        if (!version.contains('+')) {
                            return version
                        }
                        return dependencyVersions.resolveLatestVersion(
                            notation.split(':').toMutableList()
                                .also { while (it.size < 3) it.add("") }
                                .also { it[2] = version!! }
                                .joinToString(":")
                        )
                    } finally {
                        isResolving[notation] = false
                    }

                } else {
                    return version
                }

            } else {
                if (!version.contains('+')) {
                    return version
                }
            }
        }

    } else {
        version = latestVersionProperties[id]
        if (version != null && !version.contains('+')) {
            return version
        }
    }

    throw IllegalArgumentException("Predefined version can't be found: $id")
}
