package net.aihelp.core.net.mqtt.hawtdispatch;

import java.util.LinkedList;
import java.util.concurrent.Executor;

import static net.aihelp.core.net.mqtt.hawtdispatch.Dispatch.createSource;
import static net.aihelp.core.net.mqtt.hawtdispatch.Dispatch.getCurrentQueue;

/**
 * Sends runnable tasks to a DispatchQueue via a an EventAggregator
 * so that they first batch up on the sender side before being
 * sent to the DispatchQueue which then executes that tasks.
 *
 * @author <a href="http://hiramchirino.com">Hiram Chirino</a>
 */
public class AggregatingExecutor implements Executor {

    final DispatchQueue queue;
    final CustomDispatchSource<Runnable, LinkedList<Runnable>> source;

    public AggregatingExecutor(DispatchQueue queue) {
        this.queue = queue;
        this.source = createSource(EventAggregators.<Runnable>linkedList(), queue);
        this.source.setEventHandler(new Task() {
            public void run() {
                for (Runnable runnable: source.getData() ) {
                    try {
                        runnable.run();
                    } catch (Exception e) {
                        Thread thread = Thread.currentThread();
                        thread.getUncaughtExceptionHandler().uncaughtException(thread, e);
                    }
                }
            }
        });
        this.source.resume();
    }


    public void suspend() {
        source.suspend();
    }

    public void resume() {
        source.resume();
    }

    public void execute(Runnable task) {
        if (getCurrentQueue() == null) {
            queue.execute(new TaskWrapper(task));
        } else {
            source.merge(task);
        }
    }

}