package net.aihelp.core.ui.loading.indicator;

import android.animation.Animator;
import android.animation.AnimatorSet;
import android.animation.ValueAnimator;
import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.RectF;
import android.util.AttributeSet;
import android.view.View;
import android.view.animation.LinearInterpolator;
import android.widget.LinearLayout;

import net.aihelp.common.CustomConfig;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;

import androidx.annotation.Nullable;

public class AIHelpTypingIndicatorView  extends LinearLayout {

    private static final int ALPHA_LIGHT = 76;
    private static final int ALPHA_DARK = 179;
    private final long ANIMATION_DURATION;
    private final long LOOP_START_DELAY;
    Animator[] dotAnimators;
    AnimatorSet dotAnimatorSet;
    private float dotDiameter;
    private DotView[] dots;
    private int lightDotColor;
    private int interDotPadding;

    public AIHelpTypingIndicatorView(Context context) {
        this(context, null);
    }

    public AIHelpTypingIndicatorView(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public AIHelpTypingIndicatorView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        this.ANIMATION_DURATION = 900L;
        this.LOOP_START_DELAY = 450L;
        this.dotAnimators = new Animator[3];
        this.initAttributes(context, attrs);
        this.setup();
    }

    private void initAttributes(Context context, AttributeSet attrs) {
        int[] styleable = ResResolver.getStyleable("aihelp_indicator_view");
        if (styleable != null) {
            @SuppressLint("CustomViewStyleable")
            TypedArray array = getContext().obtainStyledAttributes(attrs, styleable);
            int color = array.getColor(ResResolver.getStyleableFieldIndex("aihelp_indicator_view",
                            "aihelp_dot_color"), Styles.getColor(CustomConfig.CommonSetting.textColor));
            this.lightDotColor = Color.argb(ALPHA_LIGHT, Color.red(color), Color.green(color), Color.blue(color));
            this.interDotPadding = array.getDimensionPixelSize(ResResolver.getStyleableFieldIndex("aihelp_indicator_view",
                    "aihelp_dot_padding"), Styles.dpToPx(context, 8));
            this.dotDiameter = array.getDimensionPixelSize(ResResolver.getStyleableFieldIndex("aihelp_indicator_view",
                    "aihelp_dot_diameter"), Styles.dpToPx(context, 5));
            array.recycle();
        }
    }

    protected void onWindowVisibilityChanged(int visibility) {
        super.onWindowVisibilityChanged(visibility);
        if (visibility == View.VISIBLE) {
            this.startTypingAnimation();
        } else {
            this.stopTypingAnimation();
        }

    }

    private void startTypingAnimation() {
        if (this.dotAnimatorSet == null) {
            this.dotAnimatorSet = new AnimatorSet();
            this.dotAnimatorSet.playTogether(this.dotAnimators);
            this.dotAnimatorSet.addListener(new Animator.AnimatorListener() {
                public void onAnimationStart(Animator animation) {
                }

                public void onAnimationEnd(Animator animation) {
                    animation.setStartDelay(LOOP_START_DELAY);
                    animation.start();
                }

                public void onAnimationCancel(Animator animation) {
                }

                public void onAnimationRepeat(Animator animation) {
                }
            });
            this.dotAnimatorSet.start();
        }

    }

    private void stopTypingAnimation() {
        if (this.dotAnimatorSet != null) {
            for (Animator animator : this.dotAnimatorSet.getChildAnimations()) {
                animator.cancel();
            }
            this.dotAnimatorSet.cancel();
            this.dotAnimatorSet.removeAllListeners();
            this.dotAnimatorSet = null;
            DotView[] dots = this.dots;
            for (DotView dot : dots) {
                dot.setDotColor(this.lightDotColor);
            }
        }

    }

    private void setup() {
        removeAllViews();
        this.dots = new DotView[3];

        for(int i = 0; i < 3; ++i) {
            this.dots[i] = new DotView(this.getContext(), this.lightDotColor);
            float leftPadding = this.interDotPadding / 2.0F;
            float rightPadding = this.interDotPadding / 2.0F;
            long startDelay = 0L;
            switch(i) {
                case 0:
                    leftPadding = 0.0F;
                    break;
                case 1:
                    startDelay = LOOP_START_DELAY / 2;
                    break;
                case 2:
                    rightPadding = 0.0F;
                    startDelay = LOOP_START_DELAY;
            }

            LinearLayout.LayoutParams params = new LayoutParams((int)this.dotDiameter, (int)this.dotDiameter);
            params.setMargins((int)leftPadding, 0, (int)rightPadding, 0);
            this.addView(this.dots[i], params);
            this.dotAnimators[i] = this.getAnimator(startDelay, this.dots[i]);
        }

    }

    public ValueAnimator getAnimator(long startDelay, ValueAnimator.AnimatorUpdateListener listener) {
        ValueAnimator animator = ValueAnimator.ofInt(ALPHA_LIGHT, ALPHA_DARK, ALPHA_LIGHT);
        animator.setStartDelay(startDelay);
        animator.setDuration(ANIMATION_DURATION);
        animator.setInterpolator(new LinearInterpolator());
        animator.addUpdateListener(listener);
        return animator;
    }

    public static class DotView extends View implements ValueAnimator.AnimatorUpdateListener {
        private int dotColor;
        private Paint paint;
        private float centerX;
        private float centerY;
        private float radius;
        private RectF ovalRectF;

        public DotView(Context context, int dotColor) {
            super(context);
            this.centerX = -1.0F;
            this.centerY = -1.0F;
            this.dotColor = dotColor;
            this.setup();
        }

        public DotView(Context context, @Nullable AttributeSet attrs) {
            this(context, attrs, 0);
        }

        public DotView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
            super(context, attrs, defStyleAttr);
            this.centerX = -1.0F;
            this.centerY = -1.0F;
        }

        public void setDotColor(int dotColor) {
            this.dotColor = dotColor;
            this.invalidate();
        }

        protected void onDraw(Canvas canvas) {
            canvas.drawOval(this.ovalRectF, this.paint);
        }

        protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
            super.onLayout(changed, left, top, right, bottom);
            this.centerX = (float)(this.getWidth() / 2);
            this.centerY = (float)(this.getHeight() / 2);
            this.radius = Math.min(this.centerX, this.centerY);
            this.updateOvalRectF();
        }

        private void updateOvalRectF() {
            this.ovalRectF.left = this.centerX - this.radius;
            this.ovalRectF.right = this.centerX + this.radius;
            this.ovalRectF.top = this.centerY - this.radius;
            this.ovalRectF.bottom = this.centerY + this.radius;
        }

        private void setup() {
            this.ovalRectF = new RectF();
            this.paint = new Paint();
            this.paint.setAntiAlias(true);
            this.paint.setColor(this.dotColor);
        }

        public void onAnimationUpdate(ValueAnimator animation) {
            this.dotColor = Color.argb((Integer)animation.getAnimatedValue(), Color.red(this.dotColor), Color.green(this.dotColor), Color.blue(this.dotColor));
            this.paint.setColor(this.dotColor);
            this.invalidate();
        }
    }


}

