package net.aihelp.core.util.crash;

import android.content.Context;
import android.os.Looper;
import android.util.Log;

import net.aihelp.BuildConfig;
import net.aihelp.core.util.crash.logger.AIHelpCrashLogger;

import androidx.annotation.NonNull;

public enum AIHelpCrashHandler implements OnExceptionCaughtListener {

    INSTANCE;

    private static final CharSequence AIHELP_PACKAGE_NAME = "net.aihelp";
    private static boolean sInstalled = false;

    public void init(Context context) {
        if (sInstalled) return;
        final Thread.UncaughtExceptionHandler defaultUncaughtExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
        Thread.setDefaultUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
            public void uncaughtException(@NonNull Thread thread, @NonNull Throwable throwable) {
                if (isCrashedByAIHelp(throwable) && !BuildConfig.DEBUG) {
                    onExceptionCaught(context, throwable);
                } else {
                    if (defaultUncaughtExceptionHandler != null) {
                        defaultUncaughtExceptionHandler.uncaughtException(thread, throwable);
                    }
                }
            }
        });
        sInstalled = true;
    }

    @Override
    public void onExceptionCaught(Context context, Throwable throwable) {
        AIHelpCrashLogger.fatal(throwable.getMessage(), throwable);
        if (Thread.currentThread() == Looper.getMainLooper().getThread()) {
            while (true) {
                try {
                    Looper.loop(); // restart looper to avoid crash in main thread
                } catch (Exception exception) {
                    exception.printStackTrace();
                    if (throwChoreographerException(exception)) break;
                    AIHelpCrashLogger.fatal(exception.getMessage(), exception);
                }
            }
        }
    }

    private static boolean isCrashedByAIHelp(Throwable throwable) {
        if (throwable == null) {
            return false;
        } else {
            String stackTrace = Log.getStackTraceString(throwable);
            return stackTrace.contains(AIHELP_PACKAGE_NAME);
        }
    }

    private boolean throwChoreographerException(Throwable e) {
        if (e == null) {
            return false;
        }
        StackTraceElement[] elements = e.getStackTrace();
        for (int i = elements.length - 1; i > -1; i--) {
            if (elements.length - i > 20) {
                return false;
            }
            StackTraceElement element = elements[i];
            if ("android.view.Choreographer".equals(element.getClassName())
                    && "Choreographer.java".equals(element.getFileName())
                    && "doFrame".equals(element.getMethodName())) {
                return true;
            }
        }
        return false;
    }
}
