package net.aihelp.core.util.permission;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.Build;
import android.view.View;

import net.aihelp.BuildConfig;
import net.aihelp.config.AIHelpContext;
import net.aihelp.data.attachment.AttachmentHelper;
import net.aihelp.data.attachment.AttachmentPicker;
import net.aihelp.data.attachment.IAttachmentPickerListener;
import net.aihelp.ui.webkit.AIHelpWebChromeClient;
import net.aihelp.ui.widget.snackbar.Snackbar;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.ToastUtil;

import java.lang.reflect.Method;

import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.Fragment;

public class PermissionHelper implements IPermissionCallback {

    private Activity activity;
    private Fragment fragment;
    private Object handler;
    private final String[] permissions;
    private final int requestCode;
    private final int requestType;
    private int invokeId = -1;

    private PermissionHelper(Activity activity, Fragment fragment, Object handler, String[] permissions, int requestCode, int requestType) {
        this.activity = activity;
        this.fragment = fragment;
        this.handler = handler;
        this.permissions = permissions;
        this.requestCode = requestCode;
        this.requestType = requestType;
        if (handler instanceof Activity) {
            this.activity = (Activity) handler;
        } else if (handler instanceof Fragment) {
            this.fragment = (Fragment) handler;
        }
    }

    public static PermissionHelper getInstance(Activity activity, Fragment fragment, Object handler, String[] permissions, int requestCode, int requestType) {
        if (handler == null || permissions == null || requestCode == 0) {
            if (BuildConfig.DEBUG) {
                throw new IllegalArgumentException("handler == null || permission == null || requestCode == 0!");
            }
        }
        return new PermissionHelper(activity, fragment, handler, permissions, requestCode, requestType);
    }

    public void setInvokeId(int invokeId) {
        this.invokeId = invokeId;
    }

    public boolean avoidInvoking() {
        return invokeId == requestCode;
    }

    void invokePermissionCallback(Permission.Result result) {
        if (avoidInvoking()) return;
        if (handler != null) {
            boolean isConsumed = false;
            Class<?> myClass = handler.getClass();
            while (myClass != null) {
                Method[] methods = myClass.getDeclaredMethods();
                for (Method method : methods) {
                    Permission grantedMethod = method.getAnnotation(Permission.class);
                    if (grantedMethod != null) {
                        if (requestCode == grantedMethod.requestCode()) {
                            method.setAccessible(true);
                            try {
                                method.invoke(handler, result, this, requestType);
                                isConsumed = true;
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    }
                }
                myClass = moveToSuperclass(myClass);
            }
            if (!isConsumed) {
                onPermissionResult(result);
            }
        }
    }

    private boolean onPermissionResult(Permission.Result result) {
        Activity activity = getActivity();
        if (activity != null && !activity.isFinishing()) {
            switch (result) {
                case GRANTED:
                case NONE:
                    if (fragment != null && handler instanceof IAttachmentPickerListener) {
                        AttachmentPicker.INSTANCE.setPickerHost(fragment)
                                .setAttachmentPickerListener((IAttachmentPickerListener) handler).launchPicker(requestType);
                        return true;
                    }
                    break;
                case DENIED:
                    ToastUtil.INSTANCE.showRawSnackBar(activity, ResResolver.getString("aihelp_permission_denied"), Snackbar.LENGTH_SHORT);
                    break;
                case RATIONAL:
                    ToastUtil.INSTANCE.showRawSnackBar(activity, ResResolver.getString("aihelp_permission_denied"),
                            ResResolver.getString("aihelp_yes"), Snackbar.LENGTH_INDEFINITE, new View.OnClickListener() {
                                @Override
                                public void onClick(View v) {
                                    onPermissionRational();
                                }
                            });
                    break;
                case GO_SETTING:
                    ToastUtil.INSTANCE.showRawSnackBar(activity, ResResolver.getString("aihelp_permission_ignored"),
                            ResResolver.getString("aihelp_permission_settings"), Snackbar.LENGTH_SHORT, new View.OnClickListener() {
                                @Override
                                public void onClick(View v) {
                                    onPermissionIgnored();
                                }
                            });
                    break;
                default:
                    break;
            }
        }
        return false;
    }


    private Class<?> moveToSuperclass(Class<?> clazz) {
        if (clazz == null) return null;
        clazz = clazz.getSuperclass();
        if (clazz != null) {
            String clazzName = clazz.getName();
            // Skip system classes, this degrades performance.
            // Also we might avoid some ClassNotFoundException (see FAQ for background).
            if (clazzName.startsWith("java.") || clazzName.startsWith("javax.") ||
                    clazzName.startsWith("android.") || clazzName.startsWith("androidx.")) {
                clazz = null;
            }
        }
        return clazz;
    }

    private boolean isPermissionGranted(String permission) {
        if (getActivity() != null) {
            return ContextCompat.checkSelfPermission(getActivity(), permission) == PackageManager.PERMISSION_GRANTED;
        }
        return false;
    }

    private Activity getActivity() {
        if (activity != null) {
            return activity;
        }
        if (fragment != null) {
            return fragment.getActivity();
        }
        return null;
    }

    void onRequestPermissionsResult(String[] permissions, int[] grantResults) {
        // reset the invoke id after the permission callback, so we can call the invoke method again
        setInvokeId(-1);
        for (int i = 0; i < grantResults.length; i++) {
            boolean isPermissionGranted = grantResults[i] == PackageManager.PERMISSION_GRANTED;
            if (isPermissionGranted) {
                invokePermissionCallback(Permission.Result.GRANTED);
            } else {
                if (shouldShowRequestPermissionRationale(permissions[i])) {
                    invokePermissionCallback(Permission.Result.DENIED);
                } else {
                    invokePermissionCallback(Permission.Result.GO_SETTING);
                }
            }
            // update the invoke id to requestCode again after the first loop
            // to prevent multiple callback
            setInvokeId(requestCode);
        }
    }

    Permission.State[] checkPermissionState() {
        Permission.State[] states = new Permission.State[permissions.length];
        for (int i = 0; i < permissions.length; i++) {
            String permission = permissions[i];
            Permission.State permissionState;
            if (isPermissionGranted(permission) || Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
                permissionState = Permission.State.AVAILABLE;
            } else {
                boolean hasPermissionInManifest = hasPermissionInManifest(permission);
                if (hasPermissionInManifest) {
                    permissionState = Permission.State.ASKABLE;
                    if (shouldShowRequestPermissionRationale(permission)) {
                        permissionState = Permission.State.RATIONAL;
                    }
                } else {
                    permissionState = Permission.State.UNAVAILABLE;
                }
            }
            states[i] = permissionState;
        }
        return states;
    }

    void requestPermission() {
        if (activity != null) {
            ActivityCompat.requestPermissions(activity, permissions, requestCode);
        } else if (fragment != null) {
            if (!fragment.isDetached()) {
                fragment.requestPermissions(permissions, requestCode);
            }
        }
    }

    private boolean shouldShowRequestPermissionRationale(String permission) {
        if (activity != null) {
            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.M) {
                return activity.shouldShowRequestPermissionRationale(permission);
            }
        } else if (fragment != null) {
            if (!fragment.isDetached()) {
                return fragment.shouldShowRequestPermissionRationale(permission);
            }
        }
        return false;
    }

    private static boolean hasPermissionInManifest(String permission) {
        try {
            Context context = AIHelpContext.getInstance().getContext();
            PackageInfo info = context.getPackageManager().getPackageInfo(context.getPackageName(), PackageManager.GET_PERMISSIONS);
            if (info.requestedPermissions != null) {
                for (String p : info.requestedPermissions) {
                    if (p.equals(permission)) {
                        return true;
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    private void showSettingsPage() {
        Activity activity = getActivity();
        if (activity != null) {
            try {
                Intent settingsIntent = new Intent("android.settings.APPLICATION_DETAILS_SETTINGS");
                settingsIntent.addCategory("android.intent.category.DEFAULT");
                String packageName = activity.getPackageName();
                settingsIntent.setData(Uri.parse("package:" + packageName));
                activity.startActivity(settingsIntent);
            } catch (ActivityNotFoundException var3) {
                Intent i = new Intent("android.settings.MANAGE_APPLICATIONS_SETTINGS");
                i.addCategory("android.intent.category.DEFAULT");
                activity.startActivity(i);
            }
        }
    }

    public void recycle() {
        activity = null;
        fragment = null;
        handler = null;
    }

    @Override
    public void onPermissionDenied() {
        // showSettingsPage();
    }

    @Override
    public void onPermissionRational() {
        requestPermission();
    }

    @Override
    public void onPermissionIgnored() {
        showSettingsPage();
    }

}
